﻿unit Chat;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  ByteStream, Classes, Clipbrd, Common, Contnrs, DataEntry, DataStorage, DateUtils, DIRegex,
  Graphics, JSON, PropertyList, Math, ProtoBuf, Regex, SQLite3, SysUtils, Variants, XML,
  Artifact_Utils      in 'Common\Artifact_Utils.pas',
  Columns             in 'Common\Column_Arrays\Columns.pas',
  Icon_List           in 'Common\Icon_List.pas',
  Itunes_Backup_Sig   in 'Common\Itunes_Backup_Signature_Analysis.pas',
  RS_DoNotTranslate   in 'Common\RS_DoNotTranslate.pas',
  Chat_Columns        in 'Common\Column_Arrays\Arrays_Chat.pas';

type
  TDataStoreFieldArray = array of TDataStoreField;

  TSQL_FileSearch = record
    fi_Carve_Adjustment           : integer;
    fi_Carve_Footer               : string;
    fi_Carve_Header               : string;
    fi_Glob1_Search               : string;
    fi_Glob2_Search               : string;
    fi_Glob3_Search               : string;
    fi_Icon_Category              : integer;
    fi_Icon_OS                    : integer;
    fi_Icon_Program               : integer;
    fi_Name_OS                    : string;
    fi_Name_Program               : string;
    fi_Name_Program_Type          : string;
    fi_NodeByName                 : string;
    fi_Process_As                 : string;
    fi_Process_ID                 : string;
    fi_Reference_Info             : string;
    fi_Regex_Search               : string;
    fi_Rgx_Itun_Bkup_Dmn          : string;
    fi_Rgx_Itun_Bkup_Nme          : string;
    fi_RootNodeName               : string;
    fi_Signature_Parent           : string;
    fi_Signature_Sub              : string;
    fi_SQLPrimary_Tablestr        : string;
    fi_SQLStatement               : string;
    fi_SQLTables_Required         : string;
    fi_Test_Data                  : string;
  end;

  PSQL_FileSearch = ^TSQL_FileSearch;
  TgArr = array of TSQL_FileSearch;

const
  BL_DEVELOPMENT_MODE       = False;

  ARTIFACTS_DB              = 'Artifacts_Chat.db';
  CATEGORY_NAME             = 'Chat';
  PROGRAM_NAME              = 'Chat';
  SCRIPT_DESCRIPTION        = 'Extract Chat Artifacts';
  SCRIPT_NAME               = 'Chat.pas';
  // ~~~
  ATRY_EXCEPT_STR           = 'TryExcept: '; // noslz
  ANDROID                   = 'Android'; // noslz
  BL_BOOKMARK_SOURCE        = False;
  BL_PROCEED_LOGGING        = False;
  BL_USE_FLAGS              = False;
  BS                        = '\';
  CANCELED_BY_USER          = 'Canceled by user.';
  CHAR_LENGTH               = 161;
  CHAT                      = 'Chat';
  COLON                     = ':';
  COMMA                     = ',';
  CONTACTS_STR              = 'Contacts';
  CR                        = #13#10;
  DCR                       = #13#10 + #13#10;
  DUP_FILE                  = '( \d*)?$';
  FBN_ITUNES_BACKUP_DOMAIN  = 'iTunes Backup Domain'; // noslz
  FBN_ITUNES_BACKUP_NAME    = 'iTunes Backup Name'; // noslz
  FORMAT_STR                = '%-2s %-4s %-29s %-20s %-200s';
  FORMAT_STR_HDR            = '%-2s %-34s %-20s %-200s';
  GFORMAT_STR               = '%-1s %-12s %-8s %-15s %-24s %-30s %-20s';
  GROUP_NAME                = 'Artifact Analysis';
  HYPHEN                    = ' - ';
  HYPHEN_NS                 = '-';
  ICON_ANDROID              = 1017;
  ICON_IOS                  = 1062;
  IOS                       = 'iOS'; // noslz
  MAX_CARVE_SIZE            = 1024 * 50;
  MIN_CARVE_SIZE            = 0;
  OSFILENAME                = '\\?\'; // This is needed for long file names
  PAD                       = '  ';
  PIPE                      = '|'; // noslz
  PRG_OOVOO                 = 'Oovoo'; // noslz
  PROCESS_AS_CARVE          = 'PROCESSASCARVE'; // noslz
  PROCESS_AS_PLIST          = 'PROCESSASPLIST'; // noslz
  PROCESS_AS_TEXT           = 'PROCESSASTEXT'; // noslz
  PROCESS_AS_XML            = 'PROCESSASXML'; // noslz
  PROCESSALL                = 'PROCESSALL'; // noslz
  RPAD_VALUE                = 65;
  RUNNING                   = '...';
  SPACE                     = ' ';
  STR_FILES_BY_PATH         = 'Files by path';
  TRIAGE                    = 'TRIAGE'; // noslz
  TSWT                      = 'The script will terminate.';
  VERBOSE                   = False;
  WINDOWS                   = 'Windows'; // noslz

  CHT_DISCORD_CACHE_DATA_X  = 'CHT_DISCORD_CACHE_DATA_X';

  PBUFF_FIELD1 = 1;
  PBUFF_FIELD2 = 2;
  PBUFF_FIELD3 = 3;
  PBUFF_FIELD4 = 4;

var
  gArr:                           TgArr;
  gArr_ValidatedFiles_TList:      array of TList;
  gArtConnect_CatFldr:            TArtifactEntry1;
  gArtConnect_ProgFldr:           array of TArtifactConnectEntry;
  gArtifacts_SQLite:              TSQLite3Database;
  gArtifactsDataStore:            TDataStore;
  gdb_read_bl:                    boolean;
  gFileSystemDataStore:           TDataStore;
  gNumberOfSearchItems:           integer;
  gParameter_Num_StringList:      TStringList;
  gPythonDB_pth:                  string;
  gPython_SQLite:                 TSQLite3Database;
  gtick_doprocess_i64:            uint64;
  gtick_doprocess_str:            string;
  gtick_foundlist_i64:            uint64;
  gtick_foundlist_str:            string;

function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
function FileSubSignatureMatch(Entry: TEntry): boolean;
function GetFullName(Item: PSQL_FileSearch): string;
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
function RPad(const AString: string; AChars: integer): string;
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
function TestForDoProcess(ARefNum: integer): boolean;
function TotalValidatedFileCountInTLists: integer;
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);

{$IF DEFINED (ISFEXGUI)}
type
  TScriptForm = class(TObject)
  private
    frmMain: TGUIForm;
    FMemo: TGUIMemoBox;
    pnlBottom: TGUIPanel;
    btnOK: TGUIButton;
    btnCancel: TGUIButton;
  public
    ModalResult: boolean;
    constructor Create;
    function ShowModal: boolean;
    procedure OKClick(Sender: TGUIControl);
    procedure CancelClick(Sender: TGUIControl);
    procedure SetText(const Value: string);
    procedure SetCaption(const Value: string);
  end;
{$IFEND}

implementation

{$IF DEFINED (ISFEXGUI)}

constructor TScriptForm.Create;
begin
  inherited Create;
  frmMain := NewForm(nil, SCRIPT_NAME);
  frmMain.Size(500, 480);
  pnlBottom := NewPanel(frmMain, esNone);
  pnlBottom.Size(frmMain.Width, 40);
  pnlBottom.Align(caBottom);
  btnOK := NewButton(pnlBottom, 'OK');
  btnOK.Position(pnlBottom.Width - 170, 4);
  btnOK.Size(75, 25);
  btnOK.Anchor(False, True, True, True);
  btnOK.DefaultBtn := True;
  btnOK.OnClick := OKClick;
  btnCancel := NewButton(pnlBottom, 'Cancel');
  btnCancel.Position(pnlBottom.Width - 88, 4);
  btnCancel.Size(75, 25);
  btnCancel.Anchor(False, True, True, True);
  btnCancel.CancelBtn := True;
  btnCancel.OnClick := CancelClick;
  FMemo := NewMemoBox(frmMain, [eoReadOnly]); // eoNoHScroll, eoNoVScroll
  FMemo.Color := clWhite;
  FMemo.Align(caClient);
  FMemo.FontName('Courier New');
  frmMain.CenterOnParent;
  frmMain.Invalidate;
end;

procedure TScriptForm.OKClick(Sender: TGUIControl);
begin
  // Set variables for use in the main proc. Must do this before closing the main form.
  ModalResult := False;
  ModalResult := True;
  frmMain.Close;
end;

procedure TScriptForm.CancelClick(Sender: TGUIControl);
begin
  ModalResult := False;
  Progress.DisplayMessageNow := CANCELED_BY_USER;
  frmMain.Close;
end;

function TScriptForm.ShowModal: boolean;
begin
  Execute(frmMain);
  Result := ModalResult;
end;

procedure TScriptForm.SetText(const Value: string);
begin
  FMemo.Text := Value;
end;

procedure TScriptForm.SetCaption(const Value: string);
begin
  frmMain.Text := Value;
end;
{$IFEND}

// -----------------------------------------------------------------------------
// Add 40 Character Files
// -----------------------------------------------------------------------------
procedure Add40CharFiles(UniqueList: TUniqueListOfEntries);
var
  Entry: TEntry;
  FileSystemDS: TDataStore;
begin
  FileSystemDS := GetDataStore(DATASTORE_FILESYSTEM);
  if assigned(UniqueList) and assigned(FileSystemDS) then
  begin
    Entry := FileSystemDS.First;
    while assigned(Entry) and Progress.isRunning do
    begin
      begin
        if RegexMatch(Entry.EntryName, '^[0-9a-fA-F]{40}', False) and (Entry.EntryNameExt = '') then // noslz iTunes Backup files
        begin
          UniqueList.Add(Entry);
        end;
      end;
      Entry := FileSystemDS.Next;
    end;
    FreeAndNil(FileSystemDS);
  end;
end;

// -----------------------------------------------------------------------------
// Attachment Frog
// -----------------------------------------------------------------------------
procedure Attachment_Frog(aNode: TPropertyNode; aStringList: TStringList);
var
  i: integer;
  theNodeList: TObjectList;
begin
  if assigned(aNode) and Progress.isRunning then
  begin
    theNodeList := aNode.PropChildList;
    if assigned(theNodeList) then
    begin
      for i := 0 to theNodeList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        aNode := TPropertyNode(theNodeList.items[i]);
        if (aNode.PropName = 'url') or (aNode.PropName = 'type') then // noslz
          aStringList.Add(aNode.PropDisplayValue);
        Attachment_Frog(aNode, aStringList);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Blob Text
// -----------------------------------------------------------------------------
function BlobText(const strBlob: string): string;
var
  iStart: integer;
  iIdx: integer;
begin
  Result := '';
  iStart := POS('RAW_CONTENT_VALUE_ONLY_TO_BE_VISIBLE_TO_USER', strBlob);
  if iStart <= 0 then
    Exit;
  iIdx := iStart + Length('RAW_CONTENT_VALUE_ONLY_TO_BE_VISIBLE_TO_USER') + 2;
  if Ord(strBlob[iIdx + 1]) = $1 then
    Inc(iIdx, 2);
  while iIdx < Length(strBlob) - 1 do
  begin
    if (Ord(strBlob[iIdx]) = $DA) and (Ord(strBlob[iIdx + 1]) = $0) then
      break;
    Result := Result + strBlob[iIdx];
    Inc(iIdx);
  end;
end;

// -----------------------------------------------------------------------------
// Bookmark Artifact
// -----------------------------------------------------------------------------
procedure Bookmark_Artifact_Source(Entry: TEntry; name_str: string; category_str: string; bm_comment_str: string);
var
  bmDeviceEntry: TEntry;
  bmCreate: boolean;
  bmFolder: TEntry;
  device_name_str: string;
begin
  bmCreate := True;
  if assigned(Entry) then
  begin
    bmDeviceEntry := GetDeviceEntry(Entry);
    device_name_str := '';
    if assigned(bmDeviceEntry) then device_name_str := bmDeviceEntry.SaveName + BS;
    bmFolder := FindBookmarkByName('Artifacts' + BS + 'Source Files' + BS {+ device_name_str + BS} + category_str + BS + name_str, bmCreate);
    if assigned(bmFolder) then
    begin
      if not IsItemInBookmark(bmFolder, Entry) then
      begin
        AddItemsToBookmark(bmFolder, DATASTORE_FILESYSTEM, Entry, bm_comment_str);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Column Value By Name As Date Time
// -----------------------------------------------------------------------------
function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
var
  iCol: integer;
begin
  Result := 0;
  iCol := ColumnByName(Statement, copy(colConf.sql_col, 5, Length(colConf.sql_col)));
  if (iCol > -1) then
  begin
    if colConf.read_as = ftLargeInt then
    try
      Result := Int64ToDateTime_ConvertAs(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end
    else if colConf.read_as = ftFloat then
    try
      Result := GHFloatToDateTime(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Create Artifact Description Text
// -----------------------------------------------------------------------------
function Create_Artifact_Description_Text(Item: TSQL_FileSearch): string;
var
  aStringList: TStringList;

  procedure AddIntToSL(const name_str: string; aint: integer);
  begin
    if aint > 0 then
      aStringList.Add(RPad(name_str + ':', 25) + IntToStr(aint));
  end;

  procedure AddStrToSL(const name_str: string;const astr: string);
  begin
    if Trim(astr) <> '' then
      aStringList.Add(RPad(name_str + ':', 25) + astr);
  end;

begin
  Result := '';
  aStringList := TStringList.Create;
  try
    AddStrToSL('Process_ID', Item.fi_Process_ID);
    AddStrToSL('Name_Program', Item.fi_Name_Program);
    AddStrToSL('Name_Program_Type', Item.fi_Name_Program_Type);
    AddStrToSL('Name_OS', Item.fi_Name_OS);
    AddStrToSL('Process_As', Item.fi_Process_As);
    AddIntToSL('Carve_Adjustment', Item.fi_Carve_Adjustment);
    AddStrToSL('Carve_Footer', Item.fi_Carve_Footer);
    AddStrToSL('Carve_Header', Item.fi_Carve_Header);
    AddIntToSL('Icon_Category', Item.fi_Icon_Category);
    AddIntToSL('Icon_OS', Item.fi_Icon_OS);
    AddIntToSL('Icon_Program', Item.fi_Icon_Program);
    AddStrToSL('NodeByName', Item.fi_NodeByName);
    AddStrToSL('Reference_Info', Item.fi_Reference_Info);
    AddStrToSL('Rgx_Itun_Bkup_Dmn', Item.fi_Rgx_Itun_Bkup_Dmn);
    AddStrToSL('Rgx_Itun_Bkup_Nme', Item.fi_Rgx_Itun_Bkup_Nme);
    AddStrToSL('Glob1_Search', Item.fi_Glob1_Search);
    AddStrToSL('Glob2_Search', Item.fi_Glob2_Search);
    AddStrToSL('Glob3_Search', Item.fi_Glob3_Search);
    AddStrToSL('Regex_Search', Item.fi_Regex_Search);
    AddStrToSL('RootNodeName', Item.fi_RootNodeName);
    AddStrToSL('Signature_Parent', Item.fi_Signature_Parent);
    AddStrToSL('Signature_Sub', Item.fi_Signature_Sub);
    AddStrToSL('SQLStatement', Item.fi_SQLStatement);
    AddStrToSL('fi_SQLTables_Required', Item.fi_SQLTables_Required);
    AddStrToSL('SQLPrimary_Tablestr', Item.fi_SQLPrimary_Tablestr);
    Result := aStringList.Text;
  finally
    aStringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// Create Global Search
// -----------------------------------------------------------------------------
procedure Create_Global_Search(anint: integer; Item: TSQL_FileSearch; aStringList: TStringList);
var
  Glob1_Search: string;
  Glob2_Search: string;
  Glob3_Search: string;
begin
  Glob1_Search := Item.fi_Glob1_Search;
  Glob2_Search := Item.fi_Glob2_Search;
  Glob3_Search := Item.fi_Glob3_Search;
  if Trim(Glob1_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob1_Search + ''');'); // noslz
  if Trim(Glob2_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob2_Search + ''');'); // noslz
  if Trim(Glob3_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob3_Search + ''');'); // noslz
  if Trim(Glob1_Search) =  '' then Progress.Log(IntTostr(anint) + ': No Glob Search'); // noslz
end;

// -----------------------------------------------------------------------------
// Determine Then Skip Or Add
// -----------------------------------------------------------------------------
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
var
  bm_comment_str: string;
  bmwal_Entry: TEntry;
  DeterminedFileDriverInfo: TFileTypeInformation;
  File_Added_bl: boolean;
  first_Entry: TEntry;
  i: integer;
  Item: TSQL_FileSearch;
  MatchSignature_str: string;
  NowProceed_bl: boolean;
  reason_str: string;
  Reason_StringList: TStringList;
  trunc_EntryName_str: string;

begin
  File_Added_bl := False;
  if Entry.isSystem and ((POS('$I30', Entry.EntryName) > 0) or (POS('$90', Entry.EntryName) > 0)) then
  begin
    NowProceed_bl := False;
    Exit;
  end;

  DeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
  reason_str := '';
  trunc_EntryName_str := copy(Entry.EntryName, 1, 25);
  Reason_StringList := TStringList.Create;
  try
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      NowProceed_bl := False;
      reason_str := '';
      Item := gArr[i];

      // -------------------------------------------------------------------------------
      // Special validation (these files are still sent here via the Regex match)
      // -------------------------------------------------------------------------------
      // CHT_IOS_WECHAT_MESSAGES
      if RegexMatch(Entry.EntryName, 'EnMicroMsg', False) and (not NowProceed_bl) and (UpperCase(Item.fi_Process_ID) = 'CHT_IOS_WECHAT_MESSAGES') then // noslz
      begin
        NowProceed_bl := True;
        reason_str := 'CHT_IOS_WECHAT_MESSAGES' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ')';
        Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(T)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
      end;

      // CHT_IOS_DISCORD_SQL_CACHE_DB
      if (not NowProceed_bl) and (UpperCase(Item.fi_Process_ID) = 'CHT_IOS_DISCORD_SQL_CACHE_DB') and (DeterminedFileDriverInfo.ShortDisplayName = 'URL Cache') then // noslz
      begin
        NowProceed_bl := True;
        reason_str := 'Discord Cache.db:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ')';
        Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(T)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
      end;

      // Google Takeout JSON
      if (not NowProceed_bl) and (UpperCase(Item.fi_Process_ID) = 'CHT_GOOGLE_JSON_TAKEOUT') and (DeterminedFileDriverInfo.ShortDisplayName = 'JSON') then
      begin
        NowProceed_bl := True;
        reason_str := 'Google Takeout JSON:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ')';
        Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(T)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
      end;

      // Proceed if SubDriver has been identified
      if (not NowProceed_bl) then
      begin
        if Item.fi_Signature_Sub <> '' then
        begin
          if RegexMatch(RemoveSpecialChars(DeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then
          begin
            NowProceed_bl := True;
            reason_str := 'ShortDisplayName = Required SubSig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + Item.fi_Signature_Sub + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(A)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;
      end;

      // Set the MatchSignature to the parent
      if (not NowProceed_bl) and not(UpperCase(Entry.Extension) = '.JSON') then
      begin
        MatchSignature_str := '';
        if Item.fi_Signature_Sub = '' then
          MatchSignature_str := UpperCase(Item.fi_Signature_Parent);
        // Proceed if SubDriver is blank, but File/Path Name and Parent Signature match
        if ((RegexMatch(Entry.EntryName, Item.fi_Regex_Search, False)) or (RegexMatch(Entry.FullPathName, Item.fi_Regex_Search, False))) and
          ((UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) or (RegexMatch(DeterminedFileDriverInfo.ShortDisplayName, MatchSignature_str, False))) then
        begin
          NowProceed_bl := True;
          reason_str := 'ShortDisplay matches Parent sig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(B)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      // Proceed if EntryName is unknown, but iTunes Domain, Name and Sig match
      if (not NowProceed_bl) then
      begin
        if RegexMatch(biTunes_Domain_str, Item.fi_Rgx_Itun_Bkup_Dmn, False) and RegexMatch(biTunes_Name_str, Item.fi_Rgx_Itun_Bkup_Nme, False) and (UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) then
        begin
          NowProceed_bl := True;
          reason_str := 'Proceed on Sig and iTunes Domain\Name:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(C)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      if NowProceed_bl then
      begin
        gArr_ValidatedFiles_TList[i].Add(Entry);
        File_Added_bl := True;

        // Bookmark Source Files
        if BL_BOOKMARK_SOURCE then
        begin
          bm_comment_str := Create_Artifact_Description_Text(gArr[i]);
          Bookmark_Artifact_Source(Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
          if assigned(gFileSystemDataStore) then
          begin
            first_Entry := gFileSystemDataStore.First;
            if assigned(first_Entry) then
            begin
              bmwal_Entry := gFileSystemDataStore.FindByPath(first_Entry, Entry.FullPathName + '-wal');
              if bmwal_Entry <> nil then
              begin
                Bookmark_Artifact_Source(bmwal_Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
              end;
            end;
          end;
        end;

        // Add Flags
        if BL_USE_FLAGS then Entry.Flags := Entry.Flags + [Flag5]; // Green Flag
      end;
    end;

    if NOT(File_Added_bl) then
      Reason_StringList.Add(format(GFORMAT_STR, ['', 'Ignored', '', IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]));

    for i := 0 to Reason_StringList.Count - 1 do
      Progress.Log(Reason_StringList[i]);

  finally
    Reason_StringList.free;
  end;
end;

//------------------------------------------------------------------------------
// Export Single File
//------------------------------------------------------------------------------
function Export_Single_File(anEntry: TEntry; export_fldr_str: string) : string;
var
  ExportEntryReader: TEntryReader;
  ExportFile: TFileStream;
  FileSaveName: string;
  FldrSaveName: string;
begin
  Result := '';
  ExportEntryReader := TEntryReader.Create;
  try
    Progress.DisplayMessageNow := 'Export File' + RUNNING;
    Progress.Log(RPad('Export File:', RPAD_VALUE) + anEntry.EntryName);
    FldrSaveName := export_fldr_str;
    FileSaveName := export_fldr_str + BS + anEntry.EntryName;
    if ForceDirectories(FldrSaveName) and DirectoryExists(FldrSaveName) then
    try
      if ExportEntryReader.OpenData(anEntry) then
      begin
        try
          ExportFile := TFileStream.Create(OSFILENAME + FileSaveName, fmOpenWrite or fmCreate);
          try
            ExportFile.CopyFrom(ExportEntryReader, anEntry.LogicalSize);
            Sleep(100);
            if FileExists(OSFILENAME + FileSaveName) then
              Result := FileSaveName;
          finally
            ExportFile.free;
            ExportFile := nil;
          end;
        except
          on e: exception do
          begin
            Progress.Log(e.message);
          end;
        end;
      end;
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end;
    Progress.incCurrentProgress;
  finally
    ExportEntryReader.free;
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
  end;
end;

// -----------------------------------------------------------------------------
// File Sub Signature Match
// -----------------------------------------------------------------------------
function FileSubSignatureMatch(Entry: TEntry): boolean;
var
  i: integer;
  param_num_int: integer;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  Item: TSQL_FileSearch;
begin
  Result := False;
  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
  begin
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      Item := gArr[i];
      if Item.fi_Signature_Sub <> '' then
      begin
        aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
        if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
        begin
          if BL_PROCEED_LOGGING then
            Progress.Log(RPad('Proceed' + HYPHEN + 'Identified by SubSig:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
          Result := True;
          break;
        end;
      end;
    end;
  end
  else
  begin
    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
    begin
      for i := 0 to gParameter_Num_StringList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        param_num_int := StrToInt(gParameter_Num_StringList[i]);
        Item := gArr[param_num_int];
        if Item.fi_Signature_Sub <> '' then
        begin
          aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
          if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
          begin
            if BL_PROCEED_LOGGING then
              Progress.Log(RPad('Proceed' + HYPHEN + 'File Sub-Signature Match:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
            Result := True;
            break;
          end;
        end;
      end
    end
  end;
end;

// -----------------------------------------------------------------------------
// Get Full Name
// -----------------------------------------------------------------------------
function GetFullName(Item: PSQL_FileSearch): string;
var
  ApplicationName: string;
  TypeName: string;
  OSName: string;
begin
  Result := '';
  ApplicationName := Item.fi_Name_Program;
  TypeName := Item.fi_Name_Program_Type;
  OSName := Item.fi_Name_OS;
  if (ApplicationName <> '') then
  begin
    if (TypeName <> '') then
      Result := format('%0:s %1:s', [ApplicationName, TypeName])
    else
      Result := ApplicationName;
  end
  else
    Result := TypeName;
  if OSName <> '' then
    Result := Result + ' ' + OSName;
end;

//function isGroupMe: boolean var aReader: TEntryReader;
//var
  //aRegEx: TRegEx;
  //h_offset, h_count: int64;

//begin
  //Result := False;
//  if (Entry.Extension = '.json') then // noslz
//  begin
//    aReader := TEntryReader.Create;
//    try
//      aRegEx := TRegEx.Create;
//      try
//        aRegEx.CaseSensitive := True; // Should be True for most binary regex terms. Must come before setting search term.
//        aRegEx.SearchTerm := 'groupme'; // noslz
//        aRegEx.Stream := aReader;
//        aRegEx.Progress := Progress;
//        if aReader.OpenData(Entry) then // Open the entry to search
//          try
//            aReader.Position := 0;
//            aRegEx.Find(h_offset, h_count);
//            while (h_offset <> -1) do
//            begin
//              Result := True;
//              Exit;
//            end;
//          except
//            Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + Entry.EntryName);
//          end;
//      finally
//        aRegEx.free;
//      end;
//    finally
//      aReader.free;
//    end;
//  end;
//end;

// -----------------------------------------------------------------------------
// Length of Array Table
// -----------------------------------------------------------------------------
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to 100 do
  begin
    if anArray[i].sql_col = '' then
      break;
    Result := i;
  end;
end;

// -----------------------------------------------------------------------------
// Node Frog
// -----------------------------------------------------------------------------
procedure Nodefrog(aNode: TPropertyNode; anint: integer; aStringList: TStringList);
var
  i: integer;
  theNodeList: TObjectList;
  pad_str: string;
begin
  pad_str := (StringOfChar(' ', anint * 3));
  if assigned(aNode) and Progress.isRunning then
  begin
    theNodeList := aNode.PropChildList;
    if assigned(theNodeList) then
    begin
      for i := 0 to theNodeList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        aNode := TPropertyNode(theNodeList.items[i]);
        aStringList.Add(RPad(pad_str + aNode.PropName, RPAD_VALUE) + aNode.PropDisplayValue);
        Nodefrog(aNode, anint + 1, aStringList);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Read Field 4
// -----------------------------------------------------------------------------
function ReadField4(var Buffer: TProtoBufObject): string;
var
  pbtag_int: integer;
  pbtag_typ_int: integer;
  pbtag_fld_int: integer;
  shorttemp: ShortInt;
begin
  Result := '';
  shorttemp := Buffer.readRawVarint32;
  pbtag_int := Buffer.ReadTag;
  while (pbtag_int <> 0) and (Progress.isRunning) do
  begin
    pbtag_typ_int := GetTagWireType(pbtag_int);
    pbtag_fld_int := GetTagFieldNumber(pbtag_int);
    if pbtag_fld_int = PBUFF_FIELD4 then
    begin
      shorttemp := Buffer.readRawVarint32;
      pbtag_int := Buffer.ReadTag;
      pbtag_typ_int := GetTagWireType(pbtag_int);
      pbtag_fld_int := GetTagFieldNumber(pbtag_int);
      if pbtag_fld_int = PBUFF_FIELD2 then
      begin
        shorttemp := Buffer.readRawVarint32;
        pbtag_int := Buffer.ReadTag;
        pbtag_typ_int := GetTagWireType(pbtag_int);
        pbtag_fld_int := GetTagFieldNumber(pbtag_int);
        if pbtag_fld_int = PBUFF_FIELD1 then
        begin
          Result := Buffer.ReadString;
        end; // PBUFF_FIELD1
      end; // PBUFF_FIELD2
    end // PBUFF_FIELD4
    else
    begin
      Buffer.skipField(pbtag_int);
    end;
    pbtag_int := Buffer.ReadTag;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// -----------------------------------------------------------------------------
// Setup Column For Folder
// -----------------------------------------------------------------------------
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
var
  col_label: string;
  col_source_created: TDataStoreField;
  col_source_file: TDataStoreField;
  col_source_modified: TDataStoreField;
  col_source_path: TDataStoreField;
  Field: TDataStoreField;
  i: integer;
  Item: TSQL_FileSearch;
  NumberOfColumns: integer;

begin
  Result := True;
  Item := gArr[aReferenceNumber];
  NumberOfColumns := ColCount;
  SetLength(col_DF, ColCount + 1);

  if assigned(anArtifactFolder) then
  begin
    for i := 1 to NumberOfColumns do
    begin
      try
        if not Progress.isRunning then
          Exit;
        Field := gArtifactsDataStore.DataFields.FieldByName(aItems[i].fex_col);
        if assigned(Field) and (Field.FieldType <> aItems[i].col_type) then
        begin
          MessageUser(SCRIPT_NAME + DCR + 'WARNING: New column: ' + DCR + aItems[i].fex_col + DCR + 'already exists as a different type. Creation skipped.');
          Result := False;
        end
        else
        begin
          col_label := '';
          col_DF[i] := gArtifactsDataStore.DataFields.Add(aItems[i].fex_col + col_label, aItems[i].col_type);
          if col_DF[i] = nil then
          begin
            MessageUser(SCRIPT_NAME + DCR + 'Cannot use a fixed field. Please contact support@getdata.com quoting the following error: ' + DCR + SCRIPT_NAME + SPACE + IntToStr(aReferenceNumber) + SPACE + aItems[i].fex_col);
            Result := False;
          end;
        end;
      except
        MessageUser(ATRY_EXCEPT_STR + 'Failed to create column');
      end;
    end;

    // Set the Source Columns --------------------------------------------------
    col_source_file := gArtifactsDataStore.DataFields.GetFieldByName('Source_Name');
    col_source_path := gArtifactsDataStore.DataFields.GetFieldByName('Source_Path');
    col_source_created := gArtifactsDataStore.DataFields.GetFieldByName('Source_Created');
    col_source_modified := gArtifactsDataStore.DataFields.GetFieldByName('Source_Modified');

    // Columns -----------------------------------------------------------------
    if Result then
    begin
      // Enables the change of column headers when switching folders - This is the order of displayed columns
      for i := 1 to NumberOfColumns do
      begin
        if not Progress.isRunning then
          break;
        if aItems[i].Show then
        begin
          // Progress.Log('Add Field Name: ' + col_DF[i].FieldName);
          anArtifactFolder.AddField(col_DF[i]);
        end;
      end;

      if (Item.fi_Process_As = 'POSTPROCESS') then
        anArtifactFolder.AddField(col_source_path)
      else
      begin
        anArtifactFolder.AddField(col_source_file);
        anArtifactFolder.AddField(col_source_path);
        anArtifactFolder.AddField(col_source_created);
        anArtifactFolder.AddField(col_source_modified);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Name
// -----------------------------------------------------------------------------
function SQLColumnByName(Statement: TSQLite3Statement; const Name: string): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Statement.ColumnCount do
  begin
    if not Progress.isRunning then
      break;
    if SameText(Statement.ColumnName(i), Name) then
    begin
      Result := i;
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Integer
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsInt(Statement: TSQLite3Statement; const Name: string): integer;
var
  iCol: integer;
begin
  Result := -1;
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnInt(iCol);
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Text
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsText(Statement: TSQLite3Statement; const Name: string): string;
var
  iCol: integer;
begin
  Result := '';
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnText(iCol);
end;

// -----------------------------------------------------------------------------
// SQL - Read Python.db
// -----------------------------------------------------------------------------
procedure SavePythonDBCode(python_name_str: string);
var
  select_str: string;
  sqlselect: TSQLite3Statement;
  dt_str: string;
  tmp_name_str: string;
  tmp_StringList: TStringList;
begin
  Progress.DisplayMessageNow := 'Opening SQLite database' + RUNNING;
  if FileExists(gPythonDB_pth) then
  begin
    Progress.Log(RPad('Found Python.db:', RPAD_VALUE) + gPythonDB_pth);
    gPython_SQLite := TSQLite3Database.Create;
    if assigned(gPython_SQLite) then
      try
        tmp_StringList := TStringList.Create;
        gPython_SQLite.Open(gPythonDB_pth);
        try
          select_str := 'SELECT * FROM Python_Scripts';
          try
            sqlselect := TSQLite3Statement.Create(gPython_SQLite, select_str);
            try
              while sqlselect.Step = SQLITE_ROW do
              begin
                if not Progress.isRunning then break;
                tmp_name_str := SQLColumnValueByNameAsText(sqlselect, 'Python_Name');
                if tmp_name_str = python_name_str then
                begin
                  tmp_StringList.Text := SQLColumnValueByNameAsText(sqlselect, 'Python_Code'); // noslz
                  dt_str := formatdatetime('yyyy-mm-dd - hh-nn-ss', now); // noslz
                  dt_str := dt_str + HYPHEN + 'Python_DB'; // noslz
                  if ForceDirectories(GetExportedDir + dt_str) then
                    tmp_StringList.SaveToFile(GetExportedDir + dt_str + BS + tmp_name_str + '.py'); // noslz
                end;
              end;
            finally
              sqlselect.free;
            end;
          except
            on e: exception do
              Progress.Log(e.message);
          end;
        finally
          gPython_SQLite.Close;
        end;
      finally
        FreeAndNil(gPython_SQLite);
        tmp_StringList.free;
      end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL - Read Artifacts DB
// -----------------------------------------------------------------------------
procedure Read_SQLite_DB();
var
  i: integer;
  sql_db_path_str: string;
  sqlselect: TSQLite3Statement;

begin
  // Locate the SQLite DB
  sql_db_path_str := GetDatabasesDir + 'Artifacts' + BS + ARTIFACTS_DB; // noslz
  if not FileExists(sql_db_path_str) then
  begin
    MessageUser('Chat' + ':' + SPACE + 'Did not locate Artifacts SQLite Database:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
    Exit;
  end;
  Progress.Log(RPad('Found Database:', RPAD_VALUE) + sql_db_path_str);

  // Open the database
  gArtifacts_SQLite := TSQLite3Database.Create;
  try
    gdb_read_bl := False;
    if FileExists(sql_db_path_str) then
    try
      gArtifacts_SQLite.Open(sql_db_path_str);
      gdb_read_bl := True;
      Progress.Log(RPad('Database Read:', RPAD_VALUE) + BoolToStr(gdb_read_bl, True));
    except
      on e: exception do
      begin
        Progress.Log(e.message);
        Exit;
      end;
    end;

    // Get the number of rows in the database as gNumberOfSearchItems
    if gdb_read_bl then
    begin
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT COUNT(*) FROM Artifact_Values');
      try
        gNumberOfSearchItems := 0;
        if sqlselect.Step = SQLITE_ROW then
          gNumberOfSearchItems := sqlselect.ColumnInt(0);
      finally
        sqlselect.free;
      end;
      Progress.Log(RPad('Database Rows:', RPAD_VALUE) + IntToStr(gNumberOfSearchItems));

      if gNumberOfSearchItems = 0 then
      begin
        MessageUser('No records were read from:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
        Exit;
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      SetLength(gArr, gNumberOfSearchItems);

      // Populate the Array with values from the database
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT * FROM Artifact_Values ORDER BY fi_Name_Program');
      try
        i := 0;
        while (sqlselect.Step = SQLITE_ROW) and (Progress.isRunning) do
        begin
          gArr[i].fi_Carve_Adjustment    := SQLColumnValueByNameAsInt(sqlselect,  'fi_Carve_Adjustment');
          gArr[i].fi_Carve_Footer        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Footer');
          gArr[i].fi_Carve_Header        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Header');
          gArr[i].fi_Glob1_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob1_Search');
          gArr[i].fi_Glob2_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob2_Search');
          gArr[i].fi_Glob3_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob3_Search');
          gArr[i].fi_Icon_Category       := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Category');
          gArr[i].fi_Icon_OS             := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_OS');
          gArr[i].fi_Icon_Program        := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Program');
          gArr[i].fi_Name_Program        := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program');
          gArr[i].fi_Name_OS             := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_OS');
          gArr[i].fi_Name_Program_Type   := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program_Type');
          gArr[i].fi_NodeByName          := SQLColumnValueByNameAsText(sqlselect, 'fi_NodeByName');
          gArr[i].fi_Process_As          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_As');
          gArr[i].fi_Process_ID          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_ID');
          gArr[i].fi_Reference_Info      := SQLColumnValueByNameAsText(sqlselect, 'fi_Reference_Info');
          gArr[i].fi_Regex_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Regex_Search');
          gArr[i].fi_Rgx_Itun_Bkup_Dmn   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Dmn');
          gArr[i].fi_Rgx_Itun_Bkup_Nme   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Nme');
          gArr[i].fi_RootNodeName        := SQLColumnValueByNameAsText(sqlselect, 'fi_RootNodeName');
          gArr[i].fi_Signature_Parent    := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Parent');
          gArr[i].fi_Signature_Sub       := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Sub');
          gArr[i].fi_SQLPrimary_Tablestr := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLPrimary_Tablestr');
          gArr[i].fi_SQLStatement        := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLStatement');
          gArr[i].fi_SQLTables_Required  := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLTables_Required');
          gArr[i].fi_Test_Data           := SQLColumnValueByNameAsText(sqlselect, 'fi_Test_Data');
          inc(i);
        end;

      finally
        sqlselect.free;
      end;

    end;

  finally
    gArtifacts_SQLite.Close;
    gArtifacts_SQLite.free;
  end;
end;

// -----------------------------------------------------------------------------
// StrippedOfNonAscii
// -----------------------------------------------------------------------------
function StrippedOfNonAscii(const str: string): string;
var
  idx, Count: integer;
begin
  SetLength(Result, Length(str));
  Count := 0;
  for idx := 1 to Length(str) do
  begin
    if ((str[idx] >= #32) and (str[idx] <= #127)) or (str[idx] in [#10, #13]) then
    begin
      Inc(Count);
      Result[Count] := str[idx];
    end;
  end;
  SetLength(Result, Count);
end;

// -----------------------------------------------------------------------------
// Test for Do Process
// -----------------------------------------------------------------------------
function TestForDoProcess(ARefNum: integer): boolean;
begin
  Result := False;

  if gArr[ARefNum].fi_Process_As = 'POSTPROCESS' then
  begin
    Result := True;
    Exit;
  end;

  if (ARefNum <= Length(gArr) - 1) and Progress.isRunning then
  begin
    if (CmdLine.Params.Indexof(IntToStr(ARefNum)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      Progress.Log(RPad('Process List #' + IntToStr(ARefNum) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ARefNum].Count) + '):', RPAD_VALUE) + gArr[ARefNum].fi_Name_Program + SPACE + gArr[ARefNum].fi_Name_Program_Type + RUNNING);
      if gArr[ArefNum].fi_Process_ID <> '' then  Progress.Log(RPad('Process ID:', RPAD_VALUE) + gArr[ARefNum].fi_Process_ID);
      if gArr[ARefNum].fi_Process_As <> '' then Progress.Log(RPad('fi_Process_As:', RPAD_VALUE) + gArr[ARefNum].fi_Process_As);
      Result := True;
    end;
  {$IF DEFINED (ISFEXGUI)}
  end
  else
  begin
    if not Progress.isRunning then
      Exit;
    Progress.Log('Error: RefNum > ' + IntToStr(Length(gArr) - 1)); // noslz
  {$IFEND}
  end;
end;

// -----------------------------------------------------------------------------
// Total Validated File Count
// -----------------------------------------------------------------------------
function TotalValidatedFileCountInTLists: integer;
var
  i: integer;
begin
  Result := 0;
  for i := 0 to Length(gArr) - 1 do
  begin
    if not Progress.isRunning then
      break;
    Result := Result + gArr_ValidatedFiles_TList[i].Count;
  end;
end;

// =============================================================================
// Do Process
// =============================================================================
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);
var
  aArtifactEntry: TEntry;
  ADDList: TList;
  aPropertyTree: TPropertyParent;
  aRootProperty: TPropertyNode;
  blob_stream: TBytesStream;
  burner_direct_bytes: Tbytes;
  carved_str: string;
  CarvedData: TByteInfo;
  CarvedEntry: TEntry;
  cbyte: integer;
  chatsync_beg_int64, chatsync_end_int64: int64;
  chatsync_char: string;
  chatsync_HdrDate_dt: TDateTime;
  chatsync_HdrDate_hex: string;
  chatsync_HdrDate_int64: int64;
  chatsync_HdrDate_str: string;
  chatsync_message_found_count: integer;
  chatsync_msg_str: string;
  chatsync_MsgDate_dt: TDateTime;
  chatsync_MsgDate_int64: int64;
  chatsync_MsgDate_str: string;
  chatsync_users_str: string;
  ChildNodeList: TObjectList;
  col_DF: TDataStoreFieldArray;
  ColCount: integer;
  cypherbyte: byte;
  Display_Name_str: string;
  DNT_sql_col: string;
  duration_int: double;
  end_pos: int64;
  end_pos_max: int64;
  FooterProgress: TPAC;
  g, h, i, j, iii, k, kk, kkk, p, q, x, y, p_int: integer;
  google_voice_blob_bytes: Tbytes;
  h_startpos, h_offset, h_count, f_offset, f_count: int64;
  HeaderReader, FooterReader, CarvedEntryReader: TEntryReader;
  HeaderRegex, FooterRegEx: TRegEx;
  hexbytes_str: string;
  instagram_array_of_str: array of string;
  instagram_direct_bytes: Tbytes;
  instagram_direct_date_dt: TDateTime;
  instagram_direct_direction_str: string;
  instagram_direct_message_str: string;
  instagram_direct_recipient_int: integer;
  instagram_direct_sender_int: integer;
  instagram_direct_StringList: TStringList;
  instagram_direct_type_str: string;
  instaProp: TPropertyParent;
  Item: PSQL_FileSearch;
  LineList: TStringList;
  test_bytes_length: integer;
  mydb: TSQLite3Database;
  newJOURNALReader, newEntryReader: TEntryReader;
  newstr: string;
  newWALReader: TEntryReader;
  Node1, Node2, Node3, NextNode, PrevNode: TPropertyNode;
  NodeList1, NodeList2: TObjectList;
  nsp_data_str: string;
  NumberOfNodes: integer;
  PropertyList, Node2_PropertyList: TObjectList;
  Re: TDIPerlRegEx;
  record_count: integer;
  records_read_int: integer;
  skip_bl: boolean;
  snapchat_message_type_str: string;
  SomeBytes: Tbytes;
  sqlselect: TSQLite3Statement;
  sql_row_count: integer;
  sSize: integer;
  stPos: integer;
  Telegram_bytes: Tbytes;
  telegram_data_str: string;
  Telegram_DT: TDateTime;
  Telegram_int64: int64;
  Telegram_msgLen: integer;
  Telegram_msgoffset: integer;
  Telegram_UserID: Dword;
  temp_day, temp_month, temp_year: string;
  temp_dt: TDateTime;
  temp_int_str: string;
  temp_flt: double;
  temp_int64: int64;
  temp_str, tempstr, test_str: string;
  temp_StringList: TStringList;
  test_bytes: Tbytes;
  theTFileDriveClass: TFileDriverClass;
  TotalFiles: int64;
  type_str: string;
  variant_Array: array of variant;
  walk_back_pos: integer;
  whatsapp_call_date: string;
  YahooConfUser_str: string;
  YahooConfUserBytes: array [1 .. 1000] of byte;
  YahooDateTime: TDateTime;
  YahooDecodedMsg: string;
  YahooDword: Dword;
  YahooMsgBytes: array [1 .. 1000] of byte;
  YahooMsgLength: Dword;
  YahooName: string;
  YahooTermLength: Dword;
  YahooType: Dword;
  YahooUser: Dword;
  yn: integer;

  Buffer: TProtoBufObject;
  pb_tag_int: integer;
  pb_tag_field_int: integer;

  tmp_telegram_int64: int64;

  procedure NullTheArray;
  var
    idx: integer;
  begin
    for idx := 1 to ColCount do
      variant_Array[idx] := null;
  end;

// ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼
  procedure AddToModule;
  var
    NEntry: TArtifactItem;
    IsAllEmpty: boolean;
  begin

    // Do not add when it is a Triage
    if CmdLine.Params.Indexof(TRIAGE) > -1 then
      Exit;

     // Check if all columns are empty
    IsAllEmpty := True;
    for g := 1 to ColCount do
    begin
      if (not VarIsEmpty(variant_Array[g])) and (not VarIsNull(variant_Array[g])) then
      begin
        IsAllEmpty := False;
        break;
      end;
    end;

    // If artifacts are found
    if not IsAllEmpty then
    begin

      // Create the Category Folder to the tree ----------------------------------
      if not assigned(gArtConnect_CatFldr) then
      begin
        gArtConnect_CatFldr := AddArtifactCategory(nil, CATEGORY_NAME, -1, gArr[1].fi_Icon_Category); { Sort index, icon }
        gArtConnect_CatFldr.Status := gArtConnect_CatFldr.Status + [dstUserCreated];
      end;

      // Create artifact sub-folder
      gArtConnect_ProgFldr[ref_num] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
      Item.fi_Name_Program,
      Item.fi_Name_Program_Type,
      Item.fi_Name_OS,
      Item.fi_Icon_Program,
      Item.fi_Icon_OS);

      // If the artifact sub-folder has been created
      if assigned(gArtConnect_ProgFldr[ref_num]) then
      begin
        // Set the status of the artifacts sub-folder
        gArtConnect_ProgFldr[ref_num].Status := gArtConnect_ProgFldr[ref_num].Status + [dstUserCreated];

        // Setup the columns of the artifact sub-folder
        SetUpColumnforFolder(ref_num, gArtConnect_ProgFldr[ref_num], col_DF, ColCount, aItems);

        // Create the new entry
        NEntry := TArtifactItem.Create;
        NEntry.SourceEntry := aArtifactEntry;
        NEntry.Parent := gArtConnect_ProgFldr[ref_num];
        NEntry.PhysicalSize := 0;
        NEntry.LogicalSize := 0;

        // Populate the columns
        try
          for g := 1 to ColCount do
          begin
            if not Progress.isRunning then
              break;

            if (VarIsNull(variant_Array[g])) or (VarIsEmpty(variant_Array[g])) then
              Continue;

            case col_DF[g].FieldType of
              ftDateTime:
                try
                  col_DF[g].AsDateTime[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftDateTime conversion');
                  end;
                end;

              ftFloat:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsFloat[NEntry] := StrToFloat(variant_Array[g]);
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftFloat conversion');
                    end;
                  end;

              ftInteger:
                try
                  if Trim(variant_Array[g]) = '' then
                    variant_Array[g] := null
                  else
                    col_DF[g].AsInteger[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftInteger conversion');
                  end;
                end;

              ftLargeInt:
                try
                  col_DF[g].AsInt64[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftLargeInt conversion');
                  end;
                end;

              ftString:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsString[NEntry] := variant_Array[g];
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftString conversion');
                    end;
                  end;

              ftBytes:
                try
                  col_DF[g].AsBytes[NEntry] := variantToArrayBytes(variant_Array[g]);
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftBytes conversion');
                  end;
                end;

            end;
          end;
        except
          on e: exception do
          begin
            Progress.Log(e.message);
          end;
        end;
        ADDList.Add(NEntry);
      end;
    end;
    NullTheArray;
  end;
// ▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲

  function Test_SQL_Tables(refnum: integer): boolean;
  var
    LineList: TStringList;
    idx: integer;
    sql_tbl_count: integer;
    sqlselect_row: TSQLite3Statement;
    s: integer;
    temp_sql_str: string;

  begin
    Result := True;

    // Table count check to eliminate corrupt SQL files with no tables
    sql_tbl_count := 0;
    try
      sqlselect := TSQLite3Statement.Create(mydb, 'SELECT name FROM sqlite_master WHERE type=''table''');
      try
        while sqlselect.Step = SQLITE_ROW do
        begin
          if not Progress.isRunning then
            break;
          sql_tbl_count := sql_tbl_count + 1;
        end;
      finally
        FreeAndNil(sqlselect);
      end;
    except
      Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
      Result := False;
      Exit;
    end;

    if sql_tbl_count = 0 then
    begin
      Result := False;
      Exit;
    end;

    // Table count check to eliminate SQL files without required tables
    if (sql_tbl_count > 0) and (gArr[refnum].fi_SQLTables_Required <> '') then
    begin
      LineList := TStringList.Create;
      LineList.Delimiter := ','; // Default, but ";" is used with some locales
      LineList.StrictDelimiter := True; // Required: strings are separated *only* by Delimiter
      LineList.Sorted := True;
      LineList.Duplicates := dupIgnore;
      try
        LineList.DelimitedText := gArr[refnum].fi_SQLTables_Required;
        if LineList.Count > 0 then
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[0]) + '''))';
          for idx := 0 to LineList.Count - 1 do
            temp_sql_str := temp_sql_str + ' or (upper(name) = upper(''' + Trim(LineList[idx]) + '''))';

          temp_sql_str := temp_sql_str + ')';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) and (sqlselect_row.ColumnInt(0) = LineList.Count) then
                sql_tbl_count := 1
              else
              begin
                Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + aArtifactEntry.EntryName, RPAD_VALUE) + 'Ignored (Found ' + IntToStr(sqlselect_row.ColumnInt(0)) + ' of ' + IntToStr(LineList.Count) + ' required tables).');
                Result := False;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            MessageUser(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
            Result := False;
          end;
        end;

        // Log the missing required tables
        for s := 0 to LineList.Count - 1 do
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[s]) + ''')))';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) then
              begin
                if sqlselect_row.ColumnInt(0) = 0 then
                begin
                  Progress.Log(RPad('', RPAD_VALUE) + HYPHEN + 'Missing table:' + SPACE + Trim(LineList[s]));
                end;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
          end;
        end;

      finally
        LineList.free;
      end;
    end;

    // Row Count the matched table
    if sql_tbl_count > 0 then
    begin
      sql_row_count := 0;
      sqlselect_row := TSQLite3Statement.Create(mydb, 'SELECT COUNT(*) FROM ' + gArr[refnum].fi_SQLPrimary_Tablestr);
      try
        while sqlselect_row.Step = SQLITE_ROW do
        begin
          sql_row_count := sqlselect_row.ColumnInt(0);
          break;
        end;
      finally
        sqlselect_row.free;
      end;
    end;

  end;

  function UnixMicrosecondsToDateTime(const Microseconds: int64): TDateTime;
  const
    UnixStartDate: TDateTime = 25569; // December 30, 1899 is day 0, January 1, 1970 is 25569
    MicrosecondsPerDay: int64 = 24 * 60 * 60 * 1000000;
  begin
    Result := Microseconds / MicrosecondsPerDay + UnixStartDate;
  end;

// Burner Cache - IOS (Node by Name Method) ----------------------------------
  procedure Process_Burner_Cache_IOS;
  var
    y: integer;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_CACHE') or (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_MESSAGES_CACHE') then
    begin
      PropertyList := Node1.PropChildList;
      if assigned(PropertyList) then
      begin
        for x := 0 to PropertyList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          Node2 := TPropertyNode(PropertyList.items[x]);
          if assigned(Node2) then
          begin
            Node2_PropertyList := Node2.PropChildList;
            for y := 0 to Node2_PropertyList.Count - 1 do
            begin
              if not Progress.isRunning then
                break;
              Node3 := TPropertyNode(Node2_PropertyList.items[y]);
              if assigned(Node3) then
              begin
                for g := 1 to ColCount do
                begin
                  if not Progress.isRunning then
                    break;
                  if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_CACHE') or (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_MESSAGES_CACHE') then
                  begin
                    if (aItems[g].sql_col = 'DNT_DATECREATED') and (Node3.PropName = 'dateCreated') then // noslz
                      try
                        temp_int_str := Node3.PropValue;
                        if Length(temp_int_str) = 13 then
                        begin
                          temp_int64 := StrToInt64(temp_int_str);
                          temp_dt := UnixTimeToDateTime(temp_int64 div 1000);
                          variant_Array[g] := temp_dt;
                        end;
                      except
                        Progress.Log(ATRY_EXCEPT_STR);
                      end;
                    if (aItems[g].sql_col = 'DNT_MESSAGETYPE') and (Node3.PropName = 'messageType') then // noslz
                      if Node3.PropValue = '1' then
                        variant_Array[g] := 'Call/Voice (1)'
                      else if Node3.PropValue = '2' then
                        variant_Array[g] := 'Text/Picture (2)'
                      else
                        variant_Array[g] := 'Unknown';
                    if (aItems[g].sql_col = 'DNT_CONTACTPHONENUMBER') and (Node3.PropName = 'contactPhoneNumber') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_USERID') and (Node3.PropName = 'userId') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_BURNERID') and (Node3.PropName = 'burnerId') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_MESSAGE') and (Node3.PropName = 'message') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_DIRECTION') and (Node3.PropName = 'direction') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_MEDIAURL') and (Node3.PropName = 'mediaUrl') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_VOICEURL') and (Node3.PropName = 'voiceUrl') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                    if (aItems[g].sql_col = 'DNT_MESSAGEDURATION') and (Node3.PropName = 'messageDuration') then // noslz
                      variant_Array[g] := Node3.PropValue; // noslz
                  end;
                end;
              end;
            end;
          end;
          AddToModule;
        end;
      end;
    end;
  end;

// ByLock Android ------------------------------------------------------------
  procedure Process_ByLock;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_BYLOCK_APPPREFFILE') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
    begin
      for g := 1 to ColCount do
      begin
        if not Progress.isRunning then
          break;
        DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
        if assigned(Node1) then
          if (UpperCase(Node1.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].col_type = ftString) then
            variant_Array[g] := Node1.PropDisplayValue;
      end;
      AddToModule;
    end;
  end;

// Discord - Node by Name Method -----------------------------------------------
  procedure Process_Discord;
  var
    username_str: string;
    userid_str: string;
    filename_str: string;
    size_str: string;
    url_str: string;

    function NodeFinder(base_node: TPropertyNode; astr: string): string;
    var
      FoundNode: TPropertyNode;
    begin
      Result := '';
      FoundNode := nil;
      FoundNode := TPropertyNode(base_node.GetFirstNodeByName(astr));
      if assigned(FoundNode) then
        Result := FoundNode.PropDisplayValue;
    end;

  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_DISCORD') then
    begin
      for x := 0 to NumberOfNodes - 1 do
      begin
        if not Progress.isRunning then
          break;
        for g := 1 to ColCount do
          variant_Array[g] := null; // Null the array
        Node1 := TPropertyNode(NodeList1.items[x]);
        PropertyList := Node1.PropChildList;
        if assigned(PropertyList) then
        begin
          for y := 0 to PropertyList.Count - 1 do
          begin
            if not Progress.isRunning then
              break;
            Node2 := TPropertyNode(PropertyList.items[y]);
            if assigned(Node2) then
            begin
              for g := 1 to ColCount do
              begin
                if not Progress.isRunning then
                  break;
                DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                if Node2.PropName = DNT_sql_col then
                begin
                  variant_Array[g] := Node2.PropDisplayValue;
                  if (UpperCase(DNT_sql_col) = 'TYPE') then
                  begin
                    // https://discord.com/developers/docs/resources/channel#message-object-message-types
                    if (variant_Array[g] = '0')       then variant_Array[g] := 'Message'
                    else if (variant_Array[g] = '3')  then variant_Array[g] := 'Call'
                    else if (variant_Array[g] = '7')  then variant_Array[g] := 'User Join'
                    else if (variant_Array[g] = '19') then variant_Array[g] := 'Reply';
                  end;
                  break;
                end;
              end;

              if Node2.PropName = 'author {}' then // noslz
              begin
                username_str := NodeFinder(Node2, 'username'); // noslz
                userid_str := NodeFinder(Node2, 'id'); // noslz
                for g := 1 to ColCount do
                begin
                  if not Progress.isRunning then
                    break;
                  DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                  if DNT_sql_col = 'username'  then variant_Array[g] := username_str; // noslz
                  if DNT_sql_col = 'author_id' then variant_Array[g] := userid_str; // noslz
                end;
              end;

              if Node2.PropName = 'attachments []' then // noslz
              begin
                filename_str := NodeFinder(Node2, 'filename'); // noslz
                size_str := NodeFinder(Node2, 'size'); // noslz
                url_str := NodeFinder(Node2, 'url'); // noslz
                for g := 1 to ColCount do
                begin
                  if not Progress.isRunning then
                    break;
                  DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                  if DNT_sql_col = 'filename' then variant_Array[g] := filename_str; // noslz
                  if DNT_sql_col = 'size'     then variant_Array[g] := size_str; // noslz
                  if DNT_sql_col = 'url'      then variant_Array[g] := url_str; // noslz
                end;
              end;
            end;
          end;
        end;
        AddToModule;
      end;
    end;
  end;

  // Discord Cache - LevelDB - Python ------------------------------------------
  procedure Process_Discord_Cache(aArtifactEntry: TEntry);
  const
    PYTHON_LAUNCHER_BAT = 'python_launcher.bat'; // noslz
    PYTHON_LAUNCH_FROM_DB = 'execute_from_db_enhanced.py'; // noslz
  var
    base_path_str: string;
    ColumnIndex_StringList: TStringList;
    cellValue: string;
    colPos: integer;
    csv_full_path_name: string;
    discord_cache_file_pth: string;
    Error_str: string;
    ExitCode_crd: cardinal;
    i: integer;
    Output_str: string;
    portablepython_path_str: string;
    theCommand_str: string;
    tmp_StringList: TStringList;
  begin
    // Set the Python.db path
    gPythonDB_pth := GetDatabasesDir + 'Python.db'; // noslz
    if not FileExists(gPythonDB_pth) then
    begin
      MessageUser(RPad('Did not locate Python DB:', RPAD_VALUE) + gPythonDB_pth);
      Exit;
    end;

    // Need a debug mode to make a backup of the Database Script each time it is run.
    if BL_DEVELOPMENT_MODE then
    begin
      SavePythonDBCode('Chat_Discord_Message_Cache');
      MessageUser('DEVELOPMENT MODE:' + SPACE + UpperCase(BoolToStr(BL_DEVELOPMENT_MODE, True)));
    end;

    // Get PortablePython path
    base_path_str := ExtractFileDir(Excludetrailingpathdelimiter(GetStartupDir));
    portablepython_path_str := base_path_str + BS + 'PortablePython'; // noslz
    if not DirectoryExists(portablepython_path_str) then
    begin
      Progress.Log(RPad('Did not locate:', RPAD_VALUE) + portablepython_path_str);
      Exit;
    end;

    // Create the python generated CSV file
    if FileExists(portablepython_path_str + BS + 'Scripts' + BS + PYTHON_LAUNCH_FROM_DB) then // noslz
    begin
      csv_full_path_name := GetExportedDir + 'Temp\Artifacts\' + CHT_DISCORD_CACHE_DATA_X + BS + CHT_DISCORD_CACHE_DATA_X + '.csv'; // noslz
      discord_cache_file_pth := Export_Single_File(aArtifactEntry, GetExportedDir + 'Temp\Artifacts\' + CHT_DISCORD_CACHE_DATA_X); // noslz
      if FileExists(discord_cache_file_pth) then
      begin
         // Launch the python code directly from the DB
         theCommand_str := 'cmd.exe /c ""' + portablepython_path_str + BS + PYTHON_LAUNCHER_BAT + '" "' +  // noslz
         portablepython_path_str + BS + 'Scripts' + BS + PYTHON_LAUNCH_FROM_DB + '" ' + // noslz
         'Chat_Discord_Message_Cache ' + // noslz
         '"' + discord_cache_file_pth + '" ' +
         '--csv-output "' + csv_full_path_name + '"'; // noslz
        CaptureConsoleOutput('', theCommand_str, Output_str, Error_str, ExitCode_crd, 0, 50000);
        if ExitCode_crd <> 0 then
        begin
          Progress.Log('CMD String:' + SPACE + theCommand_str + CR +
          'Exit Code:' + SPACE + IntToStr(ExitCode_crd) + CR +
          'Output:' + SPACE + Output_str + CR +
          'Error:' + SPACE + Error_str);
        end;
      end
      else
        Progress.Log(RPad('Export not found:', RPAD_VALUE) + discord_cache_file_pth);
    end
    else
    begin
      Progress.Log(RPad('Did not locate:', RPAD_VALUE) + portablepython_path_str + BS + PYTHON_LAUNCH_FROM_DB);
      Exit;
    end;

    // Read the python generated CSV file
    if FileExists(csv_full_path_name) then
    begin
      LineList := TStringList.Create;
      LineList.Delimiter := ',';
      LineList.QuoteChar := '"';
      LineList.StrictDelimiter := True;
      tmp_StringList := TStringList.Create;
      ColumnIndex_StringList := TStringList.Create;
      try
        try
          tmp_StringList.LoadFromFile(csv_full_path_name); // Encoded UTF8-BOM by Python
        except
          on e: exception do
          begin
            Progress.Log(e.message);
            Exit;
          end;
        end;

        if tmp_StringList.Count <= 1 then
          Exit;

        // Read the header row and store column indexes
        LineList.DelimitedText := tmp_StringList[0]; // Read the first row (header)
        for i := 0 to LineList.Count - 1 do
          ColumnIndex_StringList.Values[LineList[i]] := IntToStr(i); // Store column name with index

        for i := 1 to tmp_StringList.Count - 1 do // Skip header
        begin
          if not Progress.isRunning then
            break;
          LineList.DelimitedText := tmp_StringList[i];
          for g := 1 to ColCount do
          begin
            if not Progress.isRunning then
              break;
            DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
            colPos := StrToIntDef(ColumnIndex_StringList.Values[DNT_sql_col], -1);
            if colPos <> -1 then
            begin
              cellValue := LineList[colPos];
              variant_Array[g] := cellValue;
            end;
          end;
          AddToModule;
        end;

      finally
        LineList.free;
        tmp_StringList.free;
      end;
    end;
  end;

// Google Chat - JSON Takeout ------------------------------------------------
  procedure Process_Google_Chat_JSON_Takeout;

    procedure TakeoutNodeFrog(aNode: TPropertyNode; anint: integer; aItems: TSQL_Table_array; ColCount: integer);
    var
      DNT_sql_col: string;
      g, i: integer;
      msg_id_str: string;
      theNodeList: TObjectList;
      dt_tmp_str: string;
    begin
      if assigned(aNode) and Progress.isRunning then
      begin
        theNodeList := aNode.PropChildList;
        if assigned(theNodeList) then
        begin
          for i := 0 to theNodeList.Count - 1 do
          begin
            if not Progress.isRunning then
              break;
            msg_id_str := '';
            aNode := TPropertyNode(theNodeList.items[i]);

            if (anint = 2) and RegexMatch(aNode.PropName, '^\[\d+]  {}$', True) then // noslz
            begin
              msg_id_str := aNode.PropName;
              msg_id_str := StringReplace(msg_id_str, '{', '', [rfReplaceAll]);
              msg_id_str := StringReplace(msg_id_str, '}', '', [rfReplaceAll]);
              msg_id_str := StringReplace(msg_id_str, '[', '', [rfReplaceAll]);
              msg_id_str := StringReplace(msg_id_str, ']', '', [rfReplaceAll]);
              msg_id_str := Trim(msg_id_str);
            end;

            for g := 1 to ColCount do
            begin
              if not Progress.isRunning then
                break;
              DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

              // Message ID
              if (msg_id_str <> '') and (UpperCase(DNT_sql_col) = 'MESSAGE_ID') then // noslz
                variant_Array[g] := msg_id_str;

              // Create Date String and Date Time
              if UpperCase(aNode.PropName) = (UpperCase(DNT_sql_col)) then
              begin
                if DNT_sql_col = 'created_date' then // noslz
                begin
                  dt_tmp_str := aNode.PropDisplayValue;
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Monday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Tuesday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Wednesday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Thursday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Friday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Saturday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, 'Sunday,' + SPACE, '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, ' at', '', [rfReplaceAll]); // noslz
                  dt_tmp_str := StringReplace(dt_tmp_str, ' UTC', '', [rfReplaceAll]); // noslz
                  try
                    variant_Array[1] := VarToDateTime(dt_tmp_str);
                    variant_Array[2] := aNode.PropDisplayValue;
                  except
                    Progress.Log('exception'); // noslz
                  end;
                end
                else
                  variant_Array[g] := aNode.PropDisplayValue;
                TakeoutNodeFrog(aNode, anint + 1, aItems, ColCount);
              end;
            end;
            TakeoutNodeFrog(aNode, anint + 1, aItems, ColCount);
          end;
          if (anint = 3) then
          begin
            AddToModule;
          end;
        end;
      end;
    end;

  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_GOOGLE_JSON_TAKEOUT') then
    begin
      Progress.Log('Process_Google_Chat_JSON_Takeout: ' + IntToStr(ColCount)); // noslz
      TakeoutNodeFrog(aRootProperty, 0, aItems, ColCount);
    end;
  end;

// Grindr - Node by Name Method ----------------------------------------------
  procedure Process_Grindr;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_GRINDR_PREFERENCES') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
    begin
      for g := 1 to ColCount do
        variant_Array[g] := null; // Null the array
      for x := 0 to NumberOfNodes - 1 do
      begin
        if not Progress.isRunning then
          break;
        Node1 := TPropertyNode(NodeList1.items[x]);
        PropertyList := Node1.PropChildList;
        if assigned(PropertyList) then
          for y := 0 to PropertyList.Count - 1 do
          begin
            if not Progress.isRunning then
              break;
            Node2 := TPropertyNode(PropertyList.items[y]);
            if assigned(Node2) then
              for g := 1 to ColCount do
              begin
                if not Progress.isRunning then
                  break;
                // LOGIN_EMAIL -----------------------------
                if variant_Array[1] = null then
                  if assigned(Node2) and (UpperCase(Node2.PropDisplayValue) = UpperCase('LOGIN_EMAIL')) then
                  begin
                    variant_Array[1] := Node1.PropDisplayValue;
                    break;
                  end;
                // EXPLORER_RECENT_SEARCHES ----------------
                if variant_Array[2] = null then
                  if assigned(Node2) and (UpperCase(Node2.PropDisplayValue) = UpperCase('EXPLORE_RECENT_SEARCHES')) then
                  begin
                    variant_Array[2] := Node1.PropDisplayValue;
                    break;
                  end;
                // LONGITUDE -------------------------------
                if variant_Array[3] = null then
                  if assigned(Node2) and (UpperCase(Node2.PropDisplayValue) = UpperCase('LONGITUDE')) then
                    if y + 1 <= PropertyList.Count - 1 then
                    begin
                      Node3 := TPropertyNode(PropertyList.items[y + 1]);
                      variant_Array[3] := Node3.PropDisplayValue;
                      break;
                    end;
                // LATITUDE --------------------------------
                if variant_Array[1] = null then
                  if assigned(Node2) and (UpperCase(Node2.PropDisplayValue) = UpperCase('LATITUDE')) then
                    if y + 1 <= PropertyList.Count - 1 then
                    begin
                      Node3 := TPropertyNode(PropertyList.items[y + 1]);
                      variant_Array[4] := Node3.PropDisplayValue;
                      break;
                    end; { latitude }
              end;
          end;
      end;
      AddToModule;
    end;
  end;

// GroupMe - Node by Name Method -----------------------------------------------
  procedure Process_GroupMe;
  var
    AttachmentList: TObjectList;
    Attachment_StringList: TStringList;
    PlatformNode: TPropertyNode;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_GROUPME_JSON') then
    begin
      PlatformNode := TPropertyNode(aRootProperty.GetFirstNodeByName('platform')); // noslz
      if assigned(PlatformNode) and (UpperCase(PlatformNode.PropDisplayValue) = 'GM') then // noslz  - extra check for GroupMe JSON files
      begin
        Progress.Log('Processing GroupMe');
        Attachment_StringList := TStringList.Create;
        Attachment_StringList.Sorted := True; // Must be sorted or duplicates in the following line has no effect
        Attachment_StringList.Duplicates := dupIgnore;
        try
          for g := 1 to ColCount do
            variant_Array[g] := null; // Null the array
          for x := 0 to NumberOfNodes - 1 do
          begin
            if not Progress.isRunning then
              break;
            Attachment_StringList.Clear;
            Node1 := TPropertyNode(NodeList1.items[x]);
            PropertyList := Node1.PropChildList;
            if assigned(PropertyList) then
            begin
              for g := 1 to ColCount do
                variant_Array[g] := null; // Null the array
              for y := 0 to PropertyList.Count - 1 do
              begin
                Node2 := TPropertyNode(PropertyList.items[y]);
                if assigned(Node2) then
                begin
                  for g := 1 to ColCount do
                  begin
                    if not Progress.isRunning then
                      break;
                    DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                    // ID
                    if (UpperCase(DNT_sql_col) = 'ID') and (aItems[g].col_type = ftString) then // noslz
                      variant_Array[g] := IntToStr(x)

                    else // Attachment Count
                      if (UpperCase(DNT_sql_col) = 'ATTACHMENT_COUNT') and (Node2.PropName = 'attachments []') then // noslz
                      begin
                        AttachmentList := Node2.PropChildList;
                        if assigned(AttachmentList) and (AttachmentList.Count > 0) then
                          variant_Array[g] := IntToStr(AttachmentList.Count);
                      end

                      else // Attachment URLs
                        if (UpperCase(DNT_sql_col) = 'ATTACHMENT_URLS') and (Node2.PropName = 'attachments []') then // noslz
                        begin
                          Attachment_StringList.Clear;
                          Attachment_Frog(Node2, Attachment_StringList);
                          variant_Array[g] := Attachment_StringList.CommaText;
                        end

                        else // Column Match
                          if (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].col_type = ftString) then
                            variant_Array[g] := Node2.PropDisplayValue

                          else // Date Time
                            if (UpperCase(Node2.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].read_as = ftLargeInt) then
                              try
                                if aItems[g].col_type = ftDateTime then
                                  variant_Array[g] := UnixTimeToDateTime(StrToInt(Node2.PropDisplayValue))
                                else
                                  variant_Array[g] := StrToInt(Node2.PropDisplayValue);
                              except
                                Progress.Log(ATRY_EXCEPT_STR + 'GroupMe');
                              end;

                  end;
                end;
              end;
            end;
            AddToModule;
          end;
        finally
          Attachment_StringList.free;
        end;
      end;
    end;
  end;

// Instagram Users - Node by Name Method ---------------------------------------
  procedure Process_InstagramUsers;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_INSTAGRAM_USERS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
    begin
      tempstr := '';
      PropertyList := Node1.PropChildList;
      if assigned(PropertyList) then
        for x := 0 to PropertyList.Count - 1 do
        begin
          Node2 := TPropertyNode(PropertyList.items[x]);
          if assigned(Node2) and (Node2.PropName = 'string') and (POS('"user_info"', Node2.PropValue) > 0) then
          begin
            tempstr := Node2.PropValue;
            begin
              for g := 1 to ColCount do
              begin
                if not Progress.isRunning then
                  break;
                if aItems[g].sql_col = 'DNT_User_ID' then
                  variant_Array[g] := PerlMatch('(?<="id":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_User_Name' then
                  variant_Array[g] := PerlMatch('(?<="username":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_Full_Name' then
                  variant_Array[g] := PerlMatch('(?<="full_name":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_Profile_Picture_URL' then
                  variant_Array[g] := PerlMatch('(?<="profile_pic_url":")([^"]*)(?=",")', tempstr);
              end;
            end;
          end;
        end;
      AddToModule;
    end;
  end;

// ooVoo User - Android (Node by Name Method) ----------------------------------
  procedure Process_OovooUser_Android;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_OOVOO_USER') and (UpperCase(Item^.fi_Name_OS) = UpperCase(ANDROID)) then
    begin
      tempstr := '';
      PropertyList := Node1.PropChildList;
      if assigned(PropertyList) then
        for x := 0 to PropertyList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          Node2 := TPropertyNode(PropertyList.items[x]);
          if assigned(Node2) and (Node2.PropName = 'string') then
          begin
            tempstr := Node2.PropValue;
            begin
              for g := 1 to ColCount do
              begin
                if not Progress.isRunning then
                  break;
                if aItems[g].sql_col = 'DNT_birthday' then
                  variant_Array[g] := PerlMatch('(?<="birthday":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_displayName' then
                  variant_Array[g] := PerlMatch('(?<="displayName":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_friendCount' then
                  variant_Array[g] := PerlMatch('(?<="friendCount":)([^"]*)(?=,")', tempstr);
                if aItems[g].sql_col = 'DNT_gender' then
                  variant_Array[g] := PerlMatch('(?<="gender":")([^"]*)(?=",")', tempstr);
                if UpperCase(aItems[g].sql_col) = 'DNT_ID' then
                  variant_Array[g] := PerlMatch('(?<="id":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_phone' then
                  variant_Array[g] := PerlMatch('(?<="phone":")([^"]*)(?=",")', tempstr);
                if aItems[g].sql_col = 'DNT_profilePicURL' then
                  variant_Array[g] := PerlMatch('(?<="profilePicURL":")([^"]*)(?=",")', tempstr);
              end;
            end;
          end;
        end;
      AddToModule;
    end;
  end;

// ooVoo User - iOS (Node by Name Method) --------------------------------------
  procedure Process_OovooUser_IOS;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_OOVOO_USER') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
    begin
      for g := 1 to ColCount do
        variant_Array[g] := null; // Null the array
      for x := 0 to NumberOfNodes - 1 do
        for g := 1 to ColCount do
        begin
          if not Progress.isRunning then
            break;
          DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
          Node1 := TPropertyNode(NodeList1.items[x]);
          if DNT_sql_col = Node1.PropName then
            variant_Array[g] := Node1.PropDisplayValue;
        end;
      AddToModule;
    end;
  end;

// SnapChat User iOS - Node by Name Method -----------------------------------
  procedure Process_SnapChatUser_IOS;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_SNAPCHAT_USER') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
    begin
      for g := 1 to ColCount do
      begin
        if not Progress.isRunning then
          break;
        DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
        if assigned(Node1) then
          if (UpperCase(Node1.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].col_type = ftString) then
            variant_Array[g] := Node1.PropDisplayValue;
      end;
      AddToModule;
    end;
  end;

// procedure - WhatsAppCalls ---------------------------------------------------
  procedure Process_WhatsAppCalls;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_WHATSAPP_CALLS') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then // noslz
    begin
      skip_bl := False;
      for g := 1 to ColCount do
        variant_Array[g] := null; // Null the array
      // Outer x loop
      for x := 0 to NodeList1.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        // Initialize
        temp_day := '';
        temp_month := '';
        temp_year := '';
        whatsapp_call_date := '';
        if not Progress.isRunning then
          break;
        Node2 := TPropertyNode(NodeList1.items[x]);
        if assigned(Node2) then
        begin
          if Node2.PropDisplayValue = '$null' then
            continue;
          NodeList2 := Node2.PropChildList;
          if assigned(NodeList2) then
          begin
            // Loop - Node List 2
            for y := 0 to NodeList2.Count - 1 do
            begin
              if not Progress.isRunning then
                break;
              // Skip if Node List 2 is not Call Detail
              Node3 := TPropertyNode(NodeList2.items[0]);
              if assigned(Node3) and (Node3.PropName = 'NS.objects') or (Node3.PropName = '$classname') then
              begin
                skip_bl := True;
                continue;
              end;
              // Process Node List 2 is Call Detail
              if assigned(Node3) and (Node3.PropName = 'peerJID') then
              begin
                if not Progress.isRunning then
                  break;
                Node3 := TPropertyNode(NodeList2.items[y]);
                if assigned(Node3) then
                begin
                  if Node3.PropName = RS_DNT_incoming then
                    if Node3.PropDisplayValue = RS_DNT_false then
                      variant_Array[6] := 'Outgoing'
                    else if Node3.PropDisplayValue = RS_DNT_true then
                      variant_Array[6] := 'Incoming';
                  if Node3.PropName = RS_DNT_duration then
                    try
                      duration_int := StrToFloat(Node3.PropDisplayValue);
                      variant_Array[5] := FormatDateTime('hh:mm:ss', duration_int / secsperday);
                    except
                      Progress.Log(ATRY_EXCEPT_STR + 'Failed to convert string to hh:mm:ss');
                    end;
                  if Node3.PropName = RS_DNT_day then
                    temp_day := Node3.PropValue;
                  if Node3.PropName = RS_DNT_month then
                    temp_month := Node3.PropValue;
                  if Node3.PropName = RS_DNT_year then
                    temp_year := Node3.PropValue;
                end;
                whatsapp_call_date := temp_month + '\' + temp_day + '\' + temp_year;
                variant_Array[3] := whatsapp_call_date;
              end;
              // Process the Time node
              if (variant_Array[3] <> null) and (variant_Array[5] <> null) and (variant_Array[6] <> null) then // Don't do it if the call record was not previously found
              begin
                if assigned(Node3) and (Node3.PropName = 'NS.time') then
                begin
                  variant_Array[1] := MacAbsoluteTimeAsDoubleToDateTime(StrToFloat(Node3.PropDisplayValue)); // Converted Time
                  variant_Array[2] := (Node3.PropDisplayValue); // RAW Time
                end
                else
                  continue;
              end
              else
                continue;
            end;
            if skip_bl then
              continue;
          end;
          // THIS IS THE CALL AT NODE 2 LEVEL
          if (POS('whatsapp', Node2.PropValue) > 0) then
            variant_Array[4] := (Node2.PropDisplayValue);
          // Add to module only if a value for each column was collected
          if (variant_Array[1] <> null) and (variant_Array[2] <> null) and (variant_Array[3] <> null) and (variant_Array[4] <> null) and (variant_Array[5] <> null) and (variant_Array[6] <> null) then
            AddToModule;
          for g := 1 to ColCount do
            variant_Array[g] := null; // Null the array
          continue; // Move to the next iteration of x
        end;
      end;
    end;
  end;

// procedure - WhatsApp Account ------------------------------------------------
  procedure Process_WhatsAppAccount;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_WHATSAPP_ACCOUNT') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then
    begin
      for g := 1 to ColCount do
      begin
        if not Progress.isRunning then
          break;
        DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col));
        if assigned(Node1) then
          if (UpperCase(Node1.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].col_type = ftString) then
            variant_Array[g] := Node1.PropDisplayValue;
      end;
      AddToModule;
    end;
  end;

// procedure - WhatsAppPreferences ---------------------------------------------
  procedure Process_WhatsAppPreferences;
  begin
    if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_WHATSAPP_PREFERENCES') and (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) then // noslz
    begin
      for g := 1 to ColCount do
        variant_Array[g] := null; // Null the array
      for g := 1 to ColCount do
      begin
        DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(DNT_sql_col)); // noslz
        if assigned(Node1) then
        begin
          if (UpperCase(Node1.PropName) = UpperCase(DNT_sql_col)) and (aItems[g].col_type = ftString) then
            variant_Array[g] := Node1.PropDisplayValue;
        end;
      end;
      AddToModule;
    end;
  end;

  procedure SnapChat_User_iOS_docobjects();
  var
    snapchat_guid_pos, next_pos, e1, b1, x, y: integer;
  begin
    if (DNT_sql_col = 'p') then
    begin
      test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
      if Length(test_bytes) > 0 then
      begin
        if (not VarIsNull(variant_Array[1])) and (not VarIsEmpty(variant_Array[1])) then
        begin
          snapchat_guid_pos := BytePos_Of_ASCII_Pattern(variant_Array[1], test_bytes);
          next_pos := snapchat_guid_pos;
          x := next_pos;
          for x := next_pos - 1 downto 0 do
          begin
            if (test_bytes[x] = $00) or (test_bytes[x] = $24) then
              Continue
            else
            begin
              e1 := x;
              for y := e1 downto 0 do
              begin
                if (test_bytes[y] <> $00) then
                  Continue
                else
                begin
                  b1 := y;
                  variant_Array[g] := BytesToString(test_bytes, b1 + 1, e1 - b1);
                  Exit;
                end;
              end;
            end;
          end;
        end;
      end;
    end;
  end;

// Start of Do Process =========================================================
var
  process_proceed_bl: boolean;
  temp_process_counter: integer;

begin
  if gArtifactsDataStore = nil then
    Exit;

  Item := @gArr[ref_num];
  temp_process_counter := 0;
  ColCount := LengthArrayTABLE(aItems);
  if (gArr_ValidatedFiles_TList[ref_num].Count > 0) then
  begin
    begin
      if UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_WECHAT_MESSAGES' then // noslz
        RunScript(GetScriptsDir + 'Artifacts\WeChat_Decrypt.pas', '', False); // noslz

      process_proceed_bl := True;
      if process_proceed_bl then
      begin
        SetLength(variant_Array, ColCount + 1);
        ADDList := TList.Create;
        newEntryReader := TEntryReader.Create;
        try
          Progress.Max := gArr_ValidatedFiles_TList[ref_num].Count;
          Progress.DisplayMessageNow := 'Process' + SPACE + PROGRAM_NAME + ' - ' + Item.fi_Name_OS + RUNNING;
          Progress.CurrentPosition := 1;

          // Regex Setup -------------------------------------------------------
          CarvedEntryReader := TEntryReader.Create;
          FooterProgress := TPAC.Create;
          FooterProgress.Start;
          FooterReader := TEntryReader.Create;
          FooterRegEx := TRegEx.Create;
          FooterRegEx.CaseSensitive := True;
          FooterRegEx.Progress := FooterProgress;
          FooterRegEx.SearchTerm := Item.fi_Carve_Footer;
          HeaderReader := TEntryReader.Create;
          HeaderRegex := TRegEx.Create;
          HeaderRegex.CaseSensitive := True;
          HeaderRegex.Progress := Progress;
          HeaderRegex.SearchTerm := Item.fi_Carve_Header;

          if HeaderRegex.LastError <> 0 then
          begin
            Progress.Log('HeaderRegex Error: ' + IntToStr(HeaderRegex.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          if FooterRegEx.LastError <> 0 then
          begin
            Progress.Log(RPad('!!!!!!! FOOTER REGEX ERROR !!!!!!!:', RPAD_VALUE) + IntToStr(FooterRegEx.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          TotalFiles := gArr_ValidatedFiles_TList[ref_num].Count;
          // Loop Validated Files ----------------------------------------------
          for i := 0 to gArr_ValidatedFiles_TList[ref_num].Count - 1 do { addList is freed a the end of this loop }
          begin
            if not Progress.isRunning then
              break;

            Progress.IncCurrentprogress;
            temp_process_counter := temp_process_counter + 1;
            Display_Name_str := GetFullName(Item);
            Progress.DisplayMessageNow := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + RUNNING;
            aArtifactEntry := TEntry(gArr_ValidatedFiles_TList[ref_num].items[i]);

            if assigned(aArtifactEntry) and newEntryReader.OpenData(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
            begin
              if BL_USE_FLAGS then aArtifactEntry.Flags := aArtifactEntry.Flags + [Flag8]; // Gray Flag = Process Routine

              // ===============================================================
              // CUSTOM PROCESSING (Python)
              // ===============================================================
              if Item.fi_Process_As = CHT_DISCORD_CACHE_DATA_X then
                Process_Discord_Cache(aArtifactEntry);

              // ===============================================================
              // PROCESS AS: Nodes - .PropName, .PropDisplayValue, .PropValue, .PropDataType
              // ===============================================================
              if ((UpperCase(Item.fi_Process_As) = PROCESS_AS_PLIST) or (UpperCase(Item.fi_Process_As) = PROCESS_AS_XML)) and
              ((UpperCase(Item.fi_Signature_Parent) = 'PLIST (BINARY)') or
              (UpperCase(Item.fi_Signature_Parent) = 'JSON') or
              (UpperCase(Item.fi_Signature_Parent) = 'XML')) then
              begin
                aPropertyTree := nil;
                try
                  if ProcessMetadataProperties(aArtifactEntry, aPropertyTree, newEntryReader) and assigned(aPropertyTree) then
                  begin
                    // Set Root Property (Level 0)
                    aRootProperty := aPropertyTree;
                    // Progress.Log(format('%-21s %-26s %-10s %-10s %-20s ',['Number of root child nodes:',IntToStr(aRootProperty.PropChildList.Count),'Bates:',IntToStr(aArtifactEntry.ID),aArtifactEntry.EntryName]));
                    if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                    begin
                      if not Progress.isRunning then
                        break;

                      if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_WHATSAPP_PREFERENCES') and
                      (UpperCase(Item^.fi_Name_OS) = UpperCase(IOS)) and
                      (aRootProperty.PropChildList.Count >= 1) then
                        Process_WhatsAppPreferences;

                      // Locate NodeByName
                      if CompareText(aRootProperty.PropName, Item.fi_NodeByName) = 0 then
                        Node1 := aRootProperty
                      else
                        Node1 := aRootProperty.GetFirstNodeByName(Item.fi_NodeByName);

                      if assigned(Node1) then
                      begin

                        // Count the number of nodes
                        NodeList1 := Node1.PropChildList;
                        if assigned(NodeList1) then
                          NumberOfNodes := NodeList1.Count
                        else
                          NumberOfNodes := 0;
                        Progress.Log(RPad('Number of child nodes:', RPAD_VALUE) + IntToStr(NumberOfNodes));
                        Process_Burner_Cache_IOS;
                        Process_ByLock;
                        Process_Discord;
                        Process_Google_Chat_JSON_Takeout;
                        Process_Grindr;
                        Process_GroupMe;
                        Process_InstagramUsers;
                        Process_OovooUser_Android;
                        Process_OovooUser_IOS;
                        Process_SnapChatUser_IOS;
                        Process_WhatsAppAccount;
                        Process_WhatsAppCalls;
                      end;

                    end
                    else
                      Progress.Log((format('%-39s %-10s %-10s', ['Could not open data:', '-', 'Bates: ' + IntToStr(aArtifactEntry.ID) + ' ' + aArtifactEntry.EntryName])));
                  end;
                finally
                  if assigned(aPropertyTree) then
                    FreeAndNil(aPropertyTree);
                end;
              end;

              // ===============================================================
              // PROCESS AS: SQL
              // ===============================================================
              if RegexMatch(Item.fi_Signature_Parent, RS_SIG_SQLITE, False) then
              begin
                newWALReader := GetWALReader(gFileSystemDataStore, aArtifactEntry);
                newJOURNALReader := GetJOURNALReader(gFileSystemDataStore, aArtifactEntry);
                try
                  mydb := TSQLite3Database.Create;
                  try
                    mydb.OpenStream(newEntryReader, newJOURNALReader, newWALReader);

                    if Test_SQL_Tables(ref_num) then
                    begin
                      records_read_int := 0;

                      // ***************************************************************************************************************************
                      sqlselect := TSQLite3Statement.Create(mydb, (Item.fi_SQLStatement));
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      //Progress.Log(Item.fi_SQLStatement);
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      // ***************************************************************************************************************************

                      while sqlselect.Step = SQLITE_ROW do
                      begin
                        if not Progress.isRunning then
                          break;
                        records_read_int := records_read_int + 1;

                        // Progress for large files
                        if sql_row_count > 15000 then
                        begin
                          Progress.DisplayMessages := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + SPACE +
                            IntToStr(records_read_int) + '/' + IntToStr(sql_row_count) + RUNNING;
                        end;

                        // Read the values from the SQL tables -----------------
                        for g := 1 to ColCount do
                        begin
                          if not Progress.isRunning then
                            break;
                          DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                          // Instagram Message - iOS
                          if (aItems[g].read_as = ftString) and (aItems[g].convert_as = UpperCase('Instagram')) then // noslz
                          begin
                            tempstr := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                            if aItems[g].fex_col = 'User Name' then
                              variant_Array[g] := PerlMatch('(?<="username":")([^"]*)(?=",")', tempstr);
                            if aItems[g].fex_col = 'Full Name' then
                              variant_Array[g] := PerlMatch('(?<="full_name":")([^"]*)(?=",")', tempstr);
                            if aItems[g].fex_col = 'Profile Picture URL' then
                              variant_Array[g] := PerlMatch('(?<="profile_pic_url":")([^"]*)(?=",")', tempstr);
                          end

                          else if aItems[g].read_as = ftString then
                            variant_Array[g] := ColumnValueByNameAsText(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftinteger) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftLargeInt) then
                            variant_Array[g] := ColumnValueByNameAsint64(sqlselect, DNT_sql_col)

                          else if (aItems[g].col_type = ftDateTime) then
                          try
                            if ColumnValueByNameAsDateTime(sqlselect, aItems[g]) > 0 then
                              variant_Array[g] := ColumnValueByNameAsDateTime(sqlselect, aItems[g]);
                          except
                            on e: exception do
                            begin
                              Progress.Log(e.message);
                            end;
                          end;

                          // Add the table name and row location
                          if DNT_sql_col = 'SQLLOCATION' then
                            variant_Array[g] := 'Table: ' + lowercase(Item^.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz

                          // SPECIAL CONVERSIONS FOLLOW ========================
                          if UpperCase(Item^.fi_Process_ID) = 'CHT_AND_LIFE360' then
                          begin
                            if (UpperCase(DNT_sql_col) = 'READ') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                            if (UpperCase(DNT_sql_col) = 'SENT') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                            if (UpperCase(DNT_sql_col) = 'DISMISSED') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                            if (UpperCase(DNT_sql_col) = 'DELETED') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                            if (UpperCase(DNT_sql_col) = 'FAILED_TO_SEND') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                            if (UpperCase(DNT_sql_col) = 'HAS_LOCATION') then // noslz
                            try
                              temp_str := ColumnValueByNameAsText(sqlselect, DNT_sql_col);
                              if temp_str = '1' then variant_Array[g] := 'Yes' else variant_Array[g] := 'No';
                            except
                            end;
                          end;

                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_DISCORD') then // noslz
                          begin
                            if (UpperCase(DNT_sql_col) = 'DATA') then // noslz
                            begin
                              nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                              if RegexMatch(nsp_data_str, '"type":0,', false) then // noslz
                              begin
                                for g := 1 to ColCount do // Match each column with a line in the JSON record
                                begin
                                  DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                                  if (UpperCase(DNT_sql_col) = 'TIMESTAMP')   then variant_Array[g] := Trim(PerlMatch('(?<="timestamp":")(.*)(?=",)',      nsp_data_str)); // noslz
                                  if (UpperCase(DNT_sql_col) = 'USERNAME')    then variant_Array[g] := Trim(PerlMatch('(?<="username":")(.*)(?=",)',       nsp_data_str)); // noslz
                                  if (UpperCase(DNT_sql_col) = 'CONTENT')     then variant_Array[g] := Trim(PerlMatch('(?<="content":")(.*)(?=",)',        nsp_data_str)); // noslz

                                  if (UpperCase(DNT_sql_col) = 'TYPE') then
                                  begin
                                    temp_str := Trim(PerlMatch('(?<="type":)(.*)(?=,)', nsp_data_str)); // noslz
                                    case temp_str of
                                      '0':  variant_Array[g] := 'Message';
                                      '19': variant_Array[g] := 'Sticker';
                                    end;
                                  end;

                                  if (UpperCase(DNT_sql_col) = 'AUTHOR_ID')   then variant_Array[g] := Trim(PerlMatch('(?<="author":{"id":"|0,"id":")(.*)(?=",)',   nsp_data_str)); // noslz
                                  if (UpperCase(DNT_sql_col) = 'MESSAGE_ID')  then variant_Array[g] := Trim(PerlMatch('(?<={"id":")(.*)(?=",)',            nsp_data_str)); // noslz
                                  if (UpperCase(DNT_sql_col) = 'CHANNEL_ID')  then variant_Array[g] := Trim(PerlMatch('(?<="channelId":")(.*)(?=",)',      nsp_data_str)); // noslz
                                  if (UpperCase(DNT_sql_col) = 'SQLLOCATION') then variant_Array[g] := 'Table: ' + lowercase(Item^.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz
                                end;
                                AddToModule;
                              end;
                            end;
                          end;

                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_DISCORD_SQL_CACHE_DB') then // noslz
                          begin
                            if (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then // noslz
                            begin
                              nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                              //if RegexMatch(nsp_data_str, '"type":\s?0,', false) then // noslz
                              begin
                                for g := 1 to ColCount do // Match each column with a line in the JSON record
                                begin
                                  DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));
                                  if (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then
                                    variant_Array[g] := StrippedOfNonAscii(nsp_data_str); // noslz
                                  if UpperCase(DNT_sql_col) = 'SQLLOCATION' then
                                    variant_Array[g] := 'Table: ' + lowercase(Item^.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')';
                                  //if (UpperCase(DNT_sql_col) = 'TIMESTAMP') then variant_Array[g] := Trim(PerlMatch('(?<="timestamp":\s?")(.*)(?=",)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'USERNAME') then variant_Array[g] := Trim(PerlMatch('(?<="username":\s?")(.*)(?=",)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'CONTENT') then variant_Array[g] := Trim(PerlMatch('(?<="content":\s?")(.*)(?=",)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'TYPE') then variant_Array[g] := Trim(PerlMatch('(?<="type":\s?)(.*)(?=,)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'AUTHOR_ID') then variant_Array[g] := Trim(PerlMatch('(?<="author":\s?{"id": ")(.*)(?=",)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'ID') then variant_Array[g] := Trim(PerlMatch('(?<=^{"id":\s?")(.*)(?=",)', nsp_data_str)); // noslz
                                  //if (UpperCase(DNT_sql_col) = 'CHANNEL_ID') then variant_Array[g] := Trim(PerlMatch('(?<="channel_id":\s?")(.*)(?=",)', nsp_data_str)); // noslz
                                end;
                                AddToModule;
                              end;
                            end;
                          end;

                          // Burner Contacts - iOS =============================
                          if (UpperCase(Item^.fi_Process_ID) = 'DNT_BURNER_IOS_CONTACTS') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then // If the column is Receiver_Data then get the entire blob data as bytes
                            begin
                              if (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then
                                nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                              burner_direct_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              blob_stream := TBytesStream.Create(burner_direct_bytes);
                              theTFileDriveClass := DetermineFileTypeStream(blob_stream);
                              if assigned(theTFileDriveClass) and (theTFileDriveClass.ClassName = 'TJSONDriver') then // Only process if it is a JSON blob
                              begin
                                temp_StringList := TStringList.Create;
                                try
                                  test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col); // Now get the JSON blob as text
                                  if Trim(test_str) <> '' then
                                  begin
                                    temp_StringList.Add(test_str); // Put the text blob into a stringlist
                                    if assigned(temp_StringList) and (temp_StringList.Count = 1) and (Trim(temp_StringList[0]) <> '') then
                                    begin
                                      if RegexMatch(temp_StringList[0], '"muted":', True) and RegexMatch(temp_StringList[0], '"dateCreated":', True) then
                                      begin
                                        for q := 0 to temp_StringList.Count - 1 do
                                        begin
                                          for h := 1 to ColCount do // Match each column with a line in the JSON record
                                          begin
                                            if (aItems[h].sql_col = 'DNT_DATECREATED') then // noslz
                                            begin
                                              temp_int_str := Trim(PerlMatch('(?<="dateCreated":)(.*)(?=,)', temp_StringList[q]));
                                              if Length(temp_int_str) = 13 then
                                                try
                                                  temp_int64 := StrToInt64(temp_int_str);
                                                  temp_dt := UnixTimeToDateTime(temp_int64 div 1000);
                                                  variant_Array[h] := temp_dt;
                                                except
                                                  Progress.Log(ATRY_EXCEPT_STR);
                                                end
                                              else
                                                break; // Only add records with a date
                                            end;
                                            if (aItems[h].sql_col = 'DNT_CONTACTNAME') then
                                              variant_Array[h] := PerlMatch('(?<="name":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_PHONENUMBER') then
                                              variant_Array[h] := PerlMatch('(?<="phoneNumber":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_CONTACTID') then
                                              variant_Array[h] := PerlMatch('(?<="id":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_BLOCKED') then
                                              variant_Array[h] := PerlMatch('(?<="blocked":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_NOTES') then
                                              variant_Array[h] := PerlMatch('(?<="notes":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_USERID') then
                                              variant_Array[h] := PerlMatch('(?<="userId":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_BURNERIDS') then
                                              variant_Array[h] := PerlMatch('(?<="burnerId":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_IMAGES') then
                                              variant_Array[h] := PerlMatch('(?<="images":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_VERIFIED') then
                                              variant_Array[h] := PerlMatch('(?<="verified":")(.*)(?=")', temp_StringList[q]); // noslz
                                            if (aItems[h].sql_col = 'DNT_MUTED') then
                                              variant_Array[h] := PerlMatch('(?<="muted":")(.*)(?=")', temp_StringList[q]); // noslz
                                          end;
                                          AddToModule;
                                        end;
                                      end;
                                    end;
                                  end;
                                finally
                                  temp_StringList.free;
                                end;
                              end;
                            end;
                          end;

                          // Burner Messages - iOS =============================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_MESSAGES') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then // If the column is Receiver_Data then get the entire blob data as bytes
                            begin
                              If (UpperCase(DNT_sql_col) = 'RECEIVER_DATA') then
                                nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                              burner_direct_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              blob_stream := TBytesStream.Create(burner_direct_bytes);
                              theTFileDriveClass := DetermineFileTypeStream(blob_stream);
                              if assigned(theTFileDriveClass) and (theTFileDriveClass.ClassName = 'TJSONDriver') then // Only process if it is a JSON blob
                              begin
                                temp_StringList := TStringList.Create;
                                try
                                  test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col); // Now get the JSON blob as text
                                  if Trim(test_str) <> '' then
                                  begin
                                    temp_StringList.Add(test_str); // Put the text blob into a stringlist
                                    if assigned(temp_StringList) and (temp_StringList.Count = 1) and (Trim(temp_StringList[0]) <> '') then
                                    begin
                                      if RegexMatch(temp_StringList[0], '"message":', True) then // noslz
                                      begin
                                        LineList := TStringList.Create;
                                        LineList.Delimiter := '}';
                                        LineList.StrictDelimiter := True; // More than one record in the blob
                                        try
                                        LineList.DelimitedText := temp_StringList[0]; // Now we can process each separate JSON record in the LineList
                                        for q := 0 to LineList.Count - 1 do
                                        begin
                                        if RegexMatch(LineList[0], '"dateCreated":', True) then // noslz
                                        begin
                                        for h := 1 to ColCount do // Match each column with a line in the JSON record
                                        begin
                                        if (aItems[h].sql_col = 'DNT_DATECREATED') then // noslz
                                        begin
                                        temp_int_str := Trim(PerlMatch('(?<="dateCreated":)(.*)(?=,)', LineList[q]));
                                        if Length(temp_int_str) = 13 then
                                        try
                                        temp_int64 := StrToInt64(temp_int_str);
                                        temp_dt := UnixTimeToDateTime(temp_int64 div 1000);
                                        variant_Array[h] := temp_dt;
                                        except
                                        Progress.Log(ATRY_EXCEPT_STR);
                                        end
                                        else
                                        break; // Only add records with a date
                                        end;
                                        if (aItems[h].sql_col = 'DNT_MESSAGETYPE') then
                                        begin
                                        temp_str := Trim(PerlMatch('(?<="messageType":)(.*)(?=,)', LineList[q]));
                                        if temp_str = '1' then
                                        variant_Array[h] := 'Call/Voice (1)'
                                        else if temp_str = '2' then
                                        variant_Array[h] := 'Text/Picture (2)'
                                        else
                                        variant_Array[h] := 'Unknown';
                                        end;
                                        if (aItems[h].sql_col = 'DNT_CONTACTPHONENUMBER') then
                                        variant_Array[h] := PerlMatch('(?<="contactPhoneNumber": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_USERID') then
                                        variant_Array[h] := PerlMatch('(?<="userId": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_BURNERID') then
                                        variant_Array[h] := PerlMatch('(?<="burnerId": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_MESSAGE') then
                                        variant_Array[h] := PerlMatch('(?<="message": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_DIRECTION') then
                                        variant_Array[h] := PerlMatch('(?<="direction": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_MEDIAURL') then
                                        variant_Array[h] := PerlMatch('(?<="mediaUrl": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_VOICEURL') then
                                        variant_Array[h] := PerlMatch('(?<="voiceUrl": ")(.*)(?=")', LineList[q]); // noslz
                                        if (aItems[h].sql_col = 'DNT_MESSAGEDURATION') then
                                        variant_Array[h] := PerlMatch('(?<="messageDuration": ")(.*)(?=")', LineList[q]); // noslz
                                        end;
                                        AddToModule;
                                        end;
                                        end;
                                        finally
                                        LineList.free;
                                        end;
                                      end;
                                    end;
                                  end;
                                finally
                                  temp_StringList.free;
                                end;
                              end;
                            end;
                          end;

                          // Google Voice - Android ============================
                          if (UpperCase(Item^.fi_Process_ID) = 'GOOGLE_VOICE_ANDROID') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'MESSAGE_BLOB') then // Get the blob data as bytes
                            begin
                              variant_Array[g] := '';
                              google_voice_blob_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              if Length(test_bytes) > 2 then
                              begin
                                if (test_bytes[0] = $0A) and (test_bytes[1] = $28) then
                                begin
                                  test_bytes_length := Length(test_bytes);
                                  for kk := 1 to test_bytes_length do
                                  begin
                                    if (test_bytes[kk] = $30) and (test_bytes[kk + 1] = $01) and (test_bytes[kk + 2] = $52) then
                                    begin
                                      sSize := test_bytes[kk + 3];
                                      test_str := '';
                                      for kkk := kk + 4 to (kk + 4 + sSize - 1) do
                                        test_str := test_str + chr(test_bytes[kkk]);
                                    end;
                                  end;
                                  if test_str <> '' then
                                    variant_Array[g] := test_str
                                end;
                              end;
                            end
                          end;

                          // Facebook Messenger Call Logs - Android ============
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_FACEBOOK_MESSENGER_CALLS') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'CALL_TYPE') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Audio (1)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Video (2)';

                            if (UpperCase(DNT_sql_col) = 'CALL_ROLE') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Outgoing (1)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Incoming (2)';
                          end;

                          // Google Duo - Android ============================== //https://www.stark4n6.com/2021/08/google-duo-android-ios-forensic-analysis.html
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_GOOGLE_DUO') then
                          begin
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'OUTGOING') then
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Incoming (0)'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Outgoing (1)'
                              else
                                variant_Array[g] := 'Unknown';

                            if (UpperCase(DNT_sql_col) = 'CALL_STATE') then
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Left Message (0)'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Missed Call (1)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Answered (2)'
                              else
                                variant_Array[g] := 'Unknown';

                            if (UpperCase(DNT_sql_col) = 'ACTIVITY_TYPE') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Call (1)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Note (2)'
                              else if (variant_Array[g] = '4') then
                                variant_Array[g] := 'Reaction (4)'
                              else
                                variant_Array[g] := 'Unknown';
                          end;

                          // IMO Chat - iOS ====================================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_IMO_CONTACTS') then
                          begin
                            if (variant_Array[g] = '0') then
                              variant_Array[g] := 'Message'
                          end;

                          // Instagram Direct Message - iOS ====================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_INSTAGRAM_DIRECT_MESSAGE') then // noslz
                          begin
                            instagram_direct_StringList := TStringList.Create;
                            try
                              if (DNT_sql_col = 'ARCHIVE') and (aItems[g].convert_as = 'IDIRECT') then // noslz
                              begin
                                //Inc(z);
                                // Instagram Direct As Bytes
                                instagram_direct_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);

                                instagram_direct_direction_str := '';
                                instagram_direct_message_str := '';
                                instagram_direct_recipient_int := 0;
                                instagram_direct_sender_int := 0;
                                instagram_direct_type_str := '';

                                blob_stream := TBytesStream.Create(instagram_direct_bytes); // Get the blob
                                try
                                  try
                                    // Progress.Log(RPad('Instagram Blob Stream Size:', RPAD_VALUE) + IntToStr(blob_stream.size));
                                    theTFileDriveClass := DetermineFileTypeStream(blob_stream);
                                    if assigned(theTFileDriveClass) then
                                    begin
                                      instaProp := nil;
                                      if ProcessMetadataStream(blob_stream, instaProp, theTFileDriveClass) and assigned(instaProp) then
                                      begin
                                        aRootProperty := instaProp;
                                        Nodefrog(aRootProperty, 0, instagram_direct_StringList);

                                        // Instagram Direct - Date
                                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName('NS.time'));
                                        if assigned(Node1) then
                                        begin
                                          instagram_direct_date_dt := MacAbsoluteTimeAsDoubleToDateTime(StrToFloat(Node1.PropDisplayValue));
                                        end;

                                        // Instagram Direct - Sent or Received
                                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName('NSString*senderPk')); // Sent or Received from Node
                                        if assigned(Node1) then
                                        begin
                                          instagram_direct_direction_str := Node1.PropDisplayValue; // https://seguridadyredes.wordpress.com/2019/09/23/breve-introduccion-a-las-sqlite-de-instagram/, 7 = Received, 8 = Sent
                                        end;

                                        Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName('$objects')); // Read only the $object children
                                        if assigned(Node1) then
                                        begin
                                          ChildNodeList := Node1.PropChildList;
                                          if assigned(ChildNodeList) then
                                          begin
                                          for k := 0 to ChildNodeList.Count - 1 do
                                            begin
                                              Node2 := TPropertyNode(ChildNodeList.items[k]);
                                              if Node2.PropDataType = 'UString' then
                                              begin
                                                if RegexMatch(Node2.PropDisplayValue, 'SUBTYPE_', False) then
                                                begin
                                                  instagram_direct_type_str := Node2.PropDisplayValue;
                                                  for p_int := 1 to 4 do
                                                  begin
                                                    PrevNode := TPropertyNode(ChildNodeList.items[k - p_int]);
                                                    if assigned(PrevNode) and (PrevNode.PropDataType = 'UString') then
                                                    begin
                                                      if instagram_direct_type_str = 'SUBTYPE_TEXT' then
                                                        instagram_direct_message_str := PrevNode.PropDisplayValue;
                                                    end
                                                  end;
                                                end;

                                        // Locate the ID numbers (Can be 3 or 4 numbers in a row)
                                        setlength(instagram_array_of_str, 4);
                                        if (Length(Node2.PropDisplayValue) = 35) and RegexMatch(Node2.PropDisplayValue, '^\d{35}$', False) then // Message ID = 35 digits
                                        begin
                                        instagram_array_of_str[0] := Node2.PropDisplayValue;
                                        if ChildNodeList.Count >= (k + 3) then
                                        for p := 1 to 3 do
                                        begin
                                        NextNode := TPropertyNode(ChildNodeList.items[k + p]);
                                        if RegexMatch(NextNode.PropDisplayValue, '^.{19,36}$', False) then
                                        instagram_array_of_str[1] := NextNode.PropDisplayValue; // Unknown = 19 to 36 and can be alpha-numeric
                                        if RegexMatch(NextNode.PropDisplayValue, '^\d{38,40}$', False) then
                                        instagram_array_of_str[2] := NextNode.PropDisplayValue; // Thread ID = 30 or 39 digits
                                        if RegexMatch(NextNode.PropDisplayValue, '^\d{8,12}$', False) then
                                        instagram_array_of_str[3] := NextNode.PropDisplayValue; // Instagram Sender ID = Between 8 and 12 digits
                                        end;
                                        end;
                                        end;
                                        end;
                                        end;
                                        end;
                                      end;
                                    end;
                                  finally
                                    blob_stream.free;
                                  end;
                                except
                                  Progress.Log(ATRY_EXCEPT_STR + 'Instagram Direct error');
                                end;
                              end;
                              for g := 1 to ColCount do
                                try
                                  if not Progress.isRunning then
                                    break;
                                  DNT_sql_col := aItems[g].sql_col;
                                  if UpperCase(DNT_sql_col) = 'DNT_TIMESTAMP' then
                                    variant_Array[g] := instagram_direct_date_dt;
                                  if UpperCase(DNT_sql_col) = 'DNT_MESSAGE' then
                                    variant_Array[g] := instagram_direct_message_str;
                                  if UpperCase(DNT_sql_col) = 'DNT_MESSAGE_ID' then
                                    variant_Array[g] := 'Message ID';
                                  if UpperCase(DNT_sql_col) = 'DNT_THREAD_ID' then
                                    variant_Array[g] := instagram_array_of_str[2];
                                  if UpperCase(DNT_sql_col) = 'DNT_SENDER_ID' then
                                    variant_Array[g] := instagram_array_of_str[3];
                                  if UpperCase(DNT_sql_col) = 'DNT_TYPE' then
                                    variant_Array[g] := instagram_direct_type_str; // SUBTYPE_
                                  if UpperCase(DNT_sql_col) = 'DNT_SQLLOCATION' then
                                    variant_Array[g] := 'Table: ' + lowercase(Item^.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz
                                except
                                  Progress.Log(ATRY_EXCEPT_STR + 'Error populating instagram direct variant array');
                                end;

                            finally
                              instagram_direct_StringList.free;
                            end;
                          end;

                          // Line Chat - Android ===============================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_LINE') then
                          begin
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'TYPE') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Message (1)'
                              else if (variant_Array[g] = '4') then
                                variant_Array[g] := 'Call (4)'
                              else if (variant_Array[g] = '5') then
                                variant_Array[g] := 'Sticker (5)'
                              else
                                variant_Array[g] := 'Unknown';
                          end;

                          // Messenger iOS - Blob Text =========================
                          if (DNT_sql_col = 'msg_blob') and (aItems[g].convert_as = 'BlobText') then
                            variant_Array[g] := BlobText(ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col));

                          // MSTeams Chat ======================================
                          if UpperCase(Item^.fi_Process_ID) = 'CHT_AND_MSTEAMS_MESSAGE' then
                          begin
                            if UpperCase(aItems[g].sql_col) = 'DNT_CONTENT' then // noslz
                            begin
                              variant_Array[g] := StringReplace(variant_Array[g], '<div>', '', [rfReplaceAll]); // noslz
                              variant_Array[g] := Trim(StringReplace(variant_Array[g], '</div>', '', [rfReplaceAll])); // noslz
                            end;
                          end;

                          // Slack Android - Direct Message ====================
                          if UpperCase(Item^.fi_Process_ID) = 'CHT_AND_SLACK_DIRECT_MESSAGES' then
                          begin
                            tempstr := '';
                            tempstr := ColumnValueByNameAsText(sqlselect, 'message_json');
                            if aItems[g].sql_col = 'DNT_MESSAGE' then
                              variant_Array[g] := PerlMatch('(?<="text":")([^"]*)(?=",")', tempstr);
                            if aItems[g].sql_col = 'DNT_TIMESTAMP' then
                              try
                                temp_flt := StrToFloat(PerlMatch('(?<="ts":")([^"]*)(?=",")', tempstr));
                                variant_Array[g] := GHFloatToDateTime(temp_flt, 'UNIX');
                              except
                              end;
                          end;

                          // Skype Activity ====================================
                          if UpperCase(Item^.fi_Process_ID) = 'CHT_SKYPE_ACTIVITY' then
                          begin
                            if aItems[g].sql_col = 'DNT_NSP_DATA' then
                              tempstr := variant_Array[g];
                            if aItems[g].sql_col = 'DNT_ARRIVALTIME' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="originalarrivaltime":")([^"]*)(?=",")', tempstr));
                            if aItems[g].sql_col = 'DNT_FROM' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="from":")([^"]*)(?=")', tempstr));
                            if aItems[g].sql_col = 'DNT_MESSAGETYPE' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="messagetype":")([^"]*)(?=",")', tempstr));
                            if aItems[g].sql_col = 'DNT_CONTENT' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="content":")([^"]*)(?=",")', tempstr));
                            if aItems[g].sql_col = 'DNT_DURATION' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<=\<duration\>)([^"]*)(?=\<\/duration\>)', tempstr));
                            if aItems[g].sql_col = 'DNT_CONVERSATIONID' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="conversationId":")([^"]*)(?=",")', tempstr));
                            if aItems[g].sql_col = 'DNT_CONVERSATIONLINK' then
                              variant_Array[g] := CleanUnicode(PerlMatch('(?<="conversationLink":")([^"]*)(?=",")', tempstr));
                          end;

                          // Skype Contacts ====================================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_SKYPE_CONTACTS') or (UpperCase(Item^.fi_Process_ID) = 'CHT_SKYPE_CONTACTS_V12_LIVE') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'NSP_DATA') then
                              nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                            if aItems[g].sql_col = 'DNT_MRI' then
                              variant_Array[g] := PerlMatch('(?<={"MRI":"8:)([^"]*)(?=",")', nsp_data_str);
                            if (aItems[g].sql_col = 'DNT_FETCHEDDATE') and (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftDateTime) then
                            begin
                              temp_int_str := PerlMatch('(?<="fetchedDate":)([^"]*)(?=,)', nsp_data_str);
                              try
                                if Trim(temp_int_str) <> '' then
                                begin
                                  temp_int64 := StrToInt64(temp_int_str);
                                  temp_dt := UnixTimeToDateTime(temp_int64 div 1000);
                                  variant_Array[g] := temp_dt;
                                end;
                              except
                                on e: exception do
                                begin
                                  Progress.Log(e.message);
                                  Progress.Log('Could not convert Android Skype Contact date/time.');
                                end;
                              end;
                            end;
                            if aItems[g].sql_col = 'DNT_DISPLAYNAMEOVERRIDE' then
                              variant_Array[g] := PerlMatch('(?<="DISPLAYNAMEOVERRIDE":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_THUMBURL' then
                              variant_Array[g] := PerlMatch('(?<="THUMBURL":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_BIRTHDAY' then
                              variant_Array[g] := PerlMatch('(?<="BIRTHDAY":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_GENDER' then
                              variant_Array[g] := PerlMatch('(?<="gender":")([^"]*)(?=,)', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_ISBLOCKED' then
                              variant_Array[g] := PerlMatch('(?<="isBlocked":)([^"]*)(?=,)', nsp_data_str);
                          end;

                          // Skype Messages v12 Live ---------------------------
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_SKYPE_MESSAGES_V12_LIVE') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'NSP_DATA') then
                              nsp_data_str := variant_Array[g]; // Must read this first to perlmatch the data
                            if aItems[g].sql_col = 'DNT_SKYPEID' then
                              variant_Array[g] := PerlMatch('(?<="id":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_ORIGINALARRIVALTIME' then
                              variant_Array[g] := PerlMatch('(?<="originalarrivaltime":")([^"]*)([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_MESSAGETYPE' then
                              variant_Array[g] := PerlMatch('(?<="messagetype":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_VERSION' then
                              variant_Array[g] := PerlMatch('(?<="version":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_COMPSETTIMIE' then
                              variant_Array[g] := PerlMatch('(?<="composetime":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CLIENTMESSAGEID' then
                              variant_Array[g] := PerlMatch('(?<="clientmessageid":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CONVERSATIONLINK' then
                              variant_Array[g] := PerlMatch('(?<="conversationLink":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CONTENT' then
                              variant_Array[g] := PerlMatch('(?<="content":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_TYPE' then
                              variant_Array[g] := PerlMatch('(?<="type":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CONVERSATIONID' then
                              variant_Array[g] := PerlMatch('(?<="conversationid":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_FROM' then
                              variant_Array[g] := PerlMatch('(?<="from":"}])(.*)(?=")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CUID' then
                              variant_Array[g] := PerlMatch('(?<="cuid":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CONVERSATIONID' then
                              variant_Array[g] := PerlMatch('(?<="conversationId":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CREATEDTIME' then
                              variant_Array[g] := PerlMatch('(?<="createdTime":)(.*)(?=,)', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CREATOR' then
                              variant_Array[g] := PerlMatch('(?<="creator":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_COMPOSETIME' then
                              variant_Array[g] := PerlMatch('(?<="composeTime")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_CONTENT' then
                              variant_Array[g] := PerlMatch('(?<="content":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_MESSAGETYPE' then
                              variant_Array[g] := PerlMatch('(?<="messagetype":")([^"]*)(?=",")', nsp_data_str);
                            if aItems[g].sql_col = 'DNT_ISMYMESSAGE' then
                              variant_Array[g] := PerlMatch('(?<="_isMyMessage":)(.*)(?=})', nsp_data_str);
                          end;

                          // SnapChat Android - Blob2 Text =====================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_SNAPCHAT_MESSAGE_ARROYO') then
                          begin
                            if (DNT_sql_col = 'content_type') then
                            begin
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Text'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Media'
                              else
                                variant_Array[g] := 'Unknown';
                              snapchat_message_type_str := variant_Array[g];
                            end;
                            if (DNT_sql_col = 'message_content') then
                            begin
                              if (snapchat_message_type_str = 'Text') then // noslz
                              begin
                                Buffer := TProtoBufObject.Create;
                                try
                                  test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                                  if Length(test_bytes) > 50 then
                                  begin
                                    Buffer.LoadFromBytes(test_bytes);
                                    pb_tag_int := Buffer.ReadTag;
                                    while (pb_tag_int <> 0) and (Progress.isRunning) do
                                    begin
                                      pb_tag_field_int := GetTagFieldNumber(pb_tag_int);
                                      case pb_tag_field_int of
                                        PBUFF_FIELD4:
                                        variant_Array[g] := ReadField4(Buffer);
                                      else
                                        Buffer.skipField(pb_tag_int);
                                      end;
                                      pb_tag_int := Buffer.ReadTag; // Read next Protobuf tag
                                    end;
                                  end;
                                finally
                                  Buffer.free;
                                end;
                              end
                              else
                                variant_Array[g] := '';
                            end;
                          end;

                          // SnapChat Android - Blob Text ======================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_SNAPCHAT_MESSAGE_MAIN') then
                          begin
                            if (DNT_sql_col = 'CONTENT') and (aItems[g].convert_as = 'SNAPBLOB') then
                            begin
                              test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              if Length(test_bytes) > 0 then
                              begin
                                if (test_bytes[0] = $10) then
                                begin
                                  test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                                  test_str := Trim(StrippedOfNonAscii(test_str));
                                  variant_Array[g] := test_str;
                                end
                                else
                                  variant_Array[g] := '';
                              end;
                            end;
                          end;

                          // SnapChat User iOS from docobjects =================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_SNAPCHAT_SNAPCHATTER') then
                            SnapChat_User_iOS_docobjects;

                          // Tango Direction ===================================  {Learning Android Forensics, 2015, Packt Publishing, Tango Analysis, Page 244}
                          if (UpperCase(Item^.fi_Process_ID) = 'DNT_TANGO_CHAT') then
                          begin
                            // Direction
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'DIRECTION') then
                              if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Sent'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Received'
                              else
                                variant_Array[g] := 'Unknown';
                          end;

                          if (UpperCase(Item^.fi_Process_ID) = 'DNT_TANGO_CHAT') then
                          begin
                            // Payload
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'PAYLOAD') then
                            begin
                              tempstr := variant_Array[g];
                              SomeBytes := Decode64StringToBytes(tempstr);
                              newstr := '';
                              if high(SomeBytes) > 26 then
                                for iii := (26 + low(SomeBytes)) to high(SomeBytes) do
                                  if ((SomeBytes[iii] < $20) or (SomeBytes[iii] > $7F)) then
                                  begin
                                    if (SomeBytes[iii] = $0D) or (SomeBytes[iii] = $0A) then
                                      newstr := newstr + ' ';
                                  end
                                  else
                                    newstr := newstr + chr(SomeBytes[iii]);
                              // Progress.Log(IntToStr(length(SomeBytes)));
                              Progress.Log('');
                              Progress.Log(newstr);
                              variant_Array[g] := newstr;
                            end;
                          end;

                          // Tango Type ----------------------------------------
                          if (UpperCase(Item^.fi_Process_ID) = 'DNT_TANGO_CHAT') then
                          begin
                            // Direction
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'TYPE') then
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Plain Text'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Video Message'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Audio Message'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := 'Image'
                              else if (variant_Array[g] = '4') then
                                variant_Array[g] := 'Coordinates'
                              else if (variant_Array[g] = '5') then
                                variant_Array[g] := 'Voice Call'
                              else if (variant_Array[g] = '36') then
                                variant_Array[g] := 'Missed Call';
                          end;

                          // Telegram Android - Blob Bytes =====================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_TELEGRAM') then
                          begin
                            if (DNT_sql_col = 'Data') and (aItems[g].convert_as = 'BlobBytes') then // noslz
                            begin
                              test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              if (Length(test_bytes) >= 29) and (test_bytes[0] = $11) and (test_bytes[27] = $59) then
                              begin
                                sSize := test_bytes[28];
                                test_str := '';
                                for kk := 1 to sSize do
                                begin
                                  if ((kk + 28) > Length(test_bytes)) or (test_bytes[28 + kk] = 0) then
                                    break;
                                  test_str := test_str + chr(test_bytes[28 + kk]);
                                end;
                              end
                              else
                              begin
                                test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                                test_str := PerlMatch('(?<=\x38\x5E)([\x00-\x7F]+)(?=\x00\x00)', test_str);
                                test_str := StrippedOfNonAscii(test_str);
                                if Trim(test_str) <> '' then
                                  variant_Array[g] := test_str;
                              end;
                              variant_Array[g] := test_str;
                            end; { Blob Bytes }
                          end;

                          // Telegram Android - Blob Bytes =====================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_TELEGRAM_V2') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'DATE') and (aItems[g].read_as = ftLargeInt) then // noslz
                            begin
                              tmp_telegram_int64 := 0;
                              // This date from SQLite will match in the blob directly before the string size
                              tmp_telegram_int64 := ColumnValueByNameAsint64(sqlselect, DNT_sql_col);
                            end;

                            if (DNT_sql_col = 'Data') and (aItems[g].convert_as = 'BlobBytes') then // noslz
                            begin
                              test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                              if (Length(test_bytes) > 0) and (test_bytes[0] = $E0) then
                              begin
                                telegram_data_str := '';
                                for kk := 0 to Length(test_bytes) - 4 do
                                begin
                                  if pcardinal(@test_bytes[kk])^ = tmp_telegram_int64 then
                                  begin
                                    sSize := test_bytes[kk + 4];
                                    stPos := kk + 4;
                                    if (sSize > 0) and ((stPos + sSize) < Length(test_bytes)) then
                                    begin
                                      // telegram_data_str := anEncoding.UTF8.GetString(test_bytes,kk+5,sSize);
                                      // Progress.Log(telegram_data_str);
                                      for kk := 1 to sSize do
                                        telegram_data_str := telegram_data_str + chr(test_bytes[stPos + kk]);
                                    end;
                                    break;
                                  end;
                                end;
                                variant_Array[g] := telegram_data_str;
                              end;
                            end; { Blob Bytes }

                          end;

                          // Telegram iOS Chat - Blob Bytes ====================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_TELEGRAM_T7') then
                          begin
                            if (DNT_sql_col = 'KEY1') and (aItems[g].convert_as = 'BlobBytes') then
                            begin
                              Telegram_bytes := ColumnValueByNameAsBlobBytes(sqlselect, 'KEY'); // noslz
                              // Date ------------------------------------------
                              try
                                Telegram_DT := GHFloatToDateTime(SwapInteger(pinteger(@Telegram_bytes[12])^), 'UNIX');
                                variant_Array[1] := Telegram_DT;
                              except
                                Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Converting DateTime');
                              end;
                              // Chat ID ---------------------------------------
                              Telegram_int64 := SwapInt64(pint64(@Telegram_bytes[0])^); // Read 8 in reverse then swap
                              variant_Array[2] := IntToStr(Telegram_int64);
                              // Message ID ------------------------------------
                              Telegram_int64 := SwapInt64(pint64(@Telegram_bytes[16])^); // Read 8 in reverse then swap
                              variant_Array[3] := IntToStr(Telegram_int64);
                            end;
                            // Message Text ------------------------------------
                            if (DNT_sql_col = 'VALUE') and (aItems[g].convert_as = 'BlobBytes') then
                            begin
                              Telegram_bytes := ColumnValueByNameAsBlobBytes(sqlselect, 'VALUE'); // noslz
                              Telegram_msgLen := (pcardinal(@Telegram_bytes[5])^);
                              if Telegram_msgLen = 2 then // ?
                                Telegram_msgoffset := 36
                              else
                                Telegram_msgoffset := 28;
                              Telegram_msgLen := (pcardinal(@Telegram_bytes[Telegram_msgoffset])^);
                              if (Telegram_msgLen > Length(Telegram_bytes) - Telegram_msgoffset) then
                              begin
                                Telegram_msgoffset := 36;
                                Telegram_msgLen := (pcardinal(@Telegram_bytes[Telegram_msgoffset])^);
                              end;
                              if (Telegram_msgLen > Length(Telegram_bytes) - Telegram_msgoffset) then
                                Telegram_msgLen := 0;
                              if Telegram_msgLen > 0 then
                                variant_Array[4] := (BytesToString(Telegram_bytes, Telegram_msgoffset + 4, Telegram_msgLen));
                            end;
                          end;

                          // TextNow Chat - IOS ================================
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_TEXTNOW') then
                          begin
                            test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                            if (UpperCase(DNT_sql_col) = 'ZREAD') then // noslz
                              if test_str = '1' then
                                variant_Array[g] := 'No'
                              else
                                variant_Array[g] := 'Yes';
                            if (UpperCase(DNT_sql_col) = 'ZDIRECTION') then // noslz
                            begin
                              if test_str = '1' then
                                variant_Array[g] := 'Incoming'
                              else if test_str = '2' then
                                variant_Array[g] := 'Outgoing'
                              else
                                variant_Array[g] := 'Unknown';
                            end;
                          end;

                          // Tiktok Chat - Android
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_TIKTOK') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'CONTENT') then // noslz
                            begin
                              test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                              test_str := CleanUnicode(PerlMatch('(?<="text":")(.*)(?=")', test_str)); // noslz
                              variant_Array[g] := test_str;
                            end;
                          end;

                          // Tiktok Chat - IOS
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_TIKTOK') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'HASREAD') then // noslz
                            begin
                              test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                              if test_str = '1' then
                                variant_Array[g] := 'Yes'
                              else
                                variant_Array[g] := 'No';
                            end;
                            if (UpperCase(DNT_sql_col) = 'CONTENT') then // noslz
                            begin
                              test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                              test_str := Trim(StrippedOfNonAscii(test_str));
                              if test_str <> '' then
                              begin
                                test_str := PerlMatch('(?<="text":")(.*)(?=")', test_str); // noslz
                                if Trim(test_str) <> '' then
                                  variant_Array[g] := CleanUnicode(test_str);
                              end;
                            end;
                          end;

                          // WhatsApp Android Direction ------------------------ //https://blog.group-ib.com/whatsapp_forensic_artifacts, https://www.academia.edu/31798603/Forensic_Analysis_of_WhatsApp_Messenger_on_Android_Smartphones
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_WHATSAPP_MESSAGES') then
                          begin
                            type_str := '';
                            if (UpperCase(DNT_sql_col) = 'KEY_FROM_ME') then
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Incoming (0)'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Outgoing (1)'
                              else
                                variant_Array[g] := 'Unknown';

                            if (UpperCase(DNT_sql_col) = 'MEDIA_WA_TYPE') then
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Text (0)'
                              else if (variant_Array[g] = '1') then
                                variant_Array[g] := 'Image (1)'
                              else if (variant_Array[g] = '2') then
                                variant_Array[g] := 'Audio (2)'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := 'Video (3)'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := 'Contact Card (4)'
                              else if (variant_Array[g] = '3') then
                                variant_Array[g] := 'Geo Position (5)'
                              else
                                variant_Array[g] := 'Unknown';

                            if (UpperCase(DNT_sql_col) = 'STATUS') then // https://arxiv.org/pdf/1507.07739.pdf 4.2.1
                              if (variant_Array[g] = '0') then
                                variant_Array[g] := 'Received (0)'
                              else if (variant_Array[g] = '4') then
                                variant_Array[g] := 'Waiting on the server (4)'
                              else if (variant_Array[g] = '5') then
                                variant_Array[g] := 'Received at the destination (5)'
                              else if (variant_Array[g] = '6') then
                                variant_Array[g] := 'Control Message (6)'
                              else
                                variant_Array[g] := 'Unknown';
                          end;

                          // Wire - Chat - Android
                          if (UpperCase(Item^.fi_Process_ID) = 'CHT_AND_WIRE') then
                          begin
                            if (UpperCase(DNT_sql_col) = 'CONTENT') then // noslz
                            begin
                              test_str := ColumnValueByNameAsBlobText(sqlselect, DNT_sql_col);
                              test_str := CleanUnicode(PerlMatch('(?<="content":")(.*)(?=")', test_str)); // noslz
                              variant_Array[g] := test_str;
                            end;
                          end;

                        end;
                        if
                        not(UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_MESSAGES') and
                        not(UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_BURNER_CONTACTS') and
                        not(UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_DISCORD_SQL_CACHE_DB') then
                          AddToModule;
                      end;
                      if assigned(sqlselect) then
                        sqlselect.free;
                      Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + copy(aArtifactEntry.EntryName, 1, 40), RPAD_VALUE) + format('%-5s %-5s', [IntToStr(records_read_int), '(SQL)']));
                    end
                    else
                      Progress.Log('<<<<<<<<<< Test_SQL_Tables returned FALSE >>>>>>>>>>'); // noslz

                  finally
                    FreeAndNil(mydb);
                    if assigned(newWALReader) then
                      FreeAndNil(newWALReader);
                    if assigned(newJOURNALReader) then
                      FreeAndNil(newJOURNALReader);
                  end;
                except
                  on e: exception do
                  begin
                    Progress.Log(ATRY_EXCEPT_STR + RPad('WARNING:', RPAD_VALUE) + 'ERROR DOING SQL QUERY!');
                    Progress.Log(e.message);
                  end;
                end;
              end;

              // Process SnapChat iOS User.plist
              if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_SNAPCHAT_USER_PLIST') then
              begin
                if HeaderReader.OpenData(aArtifactEntry) then
                begin
                  // Test SnapChat iOS 'User.plist' file signature
                  HeaderReader.Position := 0;
                  hexbytes_str := Trim(HeaderReader.AsHexString(4));
                  if hexbytes_str = '54 53 41 46' then // noslz
                  begin
                    carved_str := HeaderReader.AsPrintableChar(HeaderReader.Size);
                    for g := 1 to ColCount do
                    begin
                      if not Progress.isRunning then
                        break;
                      DNT_sql_col := lowercase(copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col)));
                      if DNT_sql_col = 'username' then // noslz
                        variant_Array[g] := PerlMatch('(?<=username\.\.)(.*)(?=\.\.)', carved_str); // noslz
                      if DNT_sql_col = 'userid' then // noslz
                        variant_Array[g] := PerlMatch('(?<=user_id\.\.)(.*)(?=\.\.)', carved_str); // noslz
                    end;
                    AddToModule;
                  end;
                end;
              end;

              // ===============================================================
              // PROCESS AS: REGEX CARVE
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_CARVE) then
              begin
                if HeaderReader.OpenData(aArtifactEntry) and FooterReader.OpenData(aArtifactEntry) then
                  try
                    Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                    h_startpos := 0;
                    HeaderReader.Position := 0;
                    HeaderRegex.Stream := HeaderReader;
                    FooterRegEx.Stream := FooterReader;
                    // Find the first match, h_offset returned is relative to start pos
                    HeaderRegex.Find(h_offset, h_count);
                    while h_offset <> -1 do // h_offset returned as -1 means no hit
                    begin
                      // Now look for a footer
                      FooterRegEx.Stream.Position := h_startpos + h_offset + Item.fi_Carve_Adjustment;
                      // Limit looking for the footer to our max size
                      end_pos := h_startpos + h_offset + MAX_CARVE_SIZE;
                      if end_pos >= HeaderRegex.Stream.Size then
                        end_pos := HeaderRegex.Stream.Size - 1; // Don't go past the end!
                      FooterRegEx.Stream.Size := end_pos;
                      FooterRegEx.Find(f_offset, f_count);
                      // Found a footer and the size was at least our minimum
                      if (f_offset <> -1) and (f_offset + f_count >= MIN_CARVE_SIZE) then
                      begin
                        // Footer found - Create an Entry for the data found
                        CarvedEntry := TEntry.Create;
                        // ByteInfo is data described as a list of byte runs, usually just one run
                        CarvedData := TByteInfo.Create;
                        // Point to the data of the file
                        CarvedData.ParentInfo := aArtifactEntry.DataInfo;
                        // Adds the block of data
                        CarvedData.RunLstAddPair(h_offset + h_startpos, f_offset + f_count);
                        CarvedEntry.DataInfo := CarvedData;
                        CarvedEntry.LogicalSize := CarvedData.Datasize;
                        CarvedEntry.PhysicalSize := CarvedData.Datasize;
                        begin
                          if CarvedEntryReader.OpenData(CarvedEntry) then
                          begin
                            CarvedEntryReader.Position := 0;
                            carved_str := CarvedEntryReader.AsPrintableChar(CarvedEntryReader.Size);
                            for g := 1 to ColCount do
                              variant_Array[g] := null; // Null the array
                            // Setup the PerRegex Searches
                            Re := TDIPerlRegEx.Create(nil);
                            Re.CompileOptions := [coCaseLess, coUnGreedy];
                            Re.SetSubjectStr(carved_str);

                            // Process CHT_SKYPE_CHATSYNC ======================
                            if (UpperCase(Item^.fi_Process_ID) = 'CHT_SKYPE_CHATSYNC') then
                            begin
                              Progress.Log(format(FORMAT_STR_HDR, [IntToStr(i + 1) + '.', 'Filename:', aArtifactEntry.EntryName, '', '']));
                              if assigned(aArtifactEntry.Parent.Parent) then
                              begin
                                Progress.Log(format(FORMAT_STR_HDR, ['', 'Folder Name:', aArtifactEntry.Parent.Parent.Parent.EntryName, '']));
                              end;
                              Progress.Log(format(FORMAT_STR_HDR, ['', 'Bates ID:', IntToStr(aArtifactEntry.ID), '']));

                              if HeaderReader.OpenData(aArtifactEntry) and FooterReader.OpenData(aArtifactEntry) then // Open the entry to search
                                try
                                  // Test ChatSync file signature
                                  HeaderReader.Position := 0;
                                  hexbytes_str := Trim(HeaderReader.AsHexString(5));
                                  if hexbytes_str = '73 43 64 42 07' then
                                  begin
                                    Progress.Log(format(FORMAT_STR_HDR, ['', 'ChatSync File Header Valid:', hexbytes_str, '']));
                                    // Get the date from the header
                                    HeaderReader.Position := 5;
                                    chatsync_HdrDate_hex := Trim(HeaderReader.AsHexString(4));
                                    HeaderReader.Position := 5;
                                    chatsync_HdrDate_int64 := HeaderReader.AsInteger;

                                    if DateCheck_Unix_10(chatsync_HdrDate_int64) then
                                    begin
                                      chatsync_HdrDate_dt := UnixTimeToDateTime(chatsync_HdrDate_int64);
                                      if (Trim(DateTimeToStr(chatsync_HdrDate_dt)) <> '') then
                                      begin
                                        chatsync_HdrDate_str := (DateTimeToStr(chatsync_HdrDate_dt));
                                        Progress.Log(format(FORMAT_STR_HDR, ['', 'Header Date:', chatsync_HdrDate_hex, chatsync_HdrDate_str, '']));
                                      end
                                      else
                                        chatsync_HdrDate_str := '';
                                    end
                                    else
                                    begin
                                      Progress.Log(StringOfChar('-', CHAR_LENGTH));
                                      continue;
                                    end;

                                    // Conversation Partners
                                    HeaderReader.Position := 52;
                                    chatsync_users_str := '';
                                    for j := 1 to 100 do
                                    begin
                                      chatsync_char := HeaderReader.AsPrintableChar(1);
                                      chatsync_users_str := chatsync_users_str + chatsync_char;
                                      if chatsync_char = ';' then
                                        break;
                                    end;
                                    if j < 100 then
                                      Progress.Log(format(FORMAT_STR_HDR, ['', 'Conversation Partners:', chatsync_users_str, '']));

                                    // Record Count
                                    h_startpos := 0;
                                    HeaderReader.Position := 0;
                                    HeaderRegex.Stream := HeaderReader;
                                    HeaderRegex.Find(h_offset, h_count); // Find the first match, h_offset returned is relative to start pos
                                    while h_offset <> -1 do // h_offset returned as -1 means no hit
                                      try
                                        if not Progress.isRunning then
                                        break;
                                        HeaderReader.Position := h_startpos + h_offset + 3;
                                        record_count := record_count + 1;
                                        HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match for the current file
                                      except
                                        Progress.Log(ATRY_EXCEPT_STR + RPad('!!!!!!!ERROR!!!!!!! - IN RECORD COUNT:', RPAD_VALUE) + aArtifactEntry.EntryName);
                                      end;
                                    Progress.Log(format(FORMAT_STR_HDR, ['', 'Expected Record Count:', IntToStr(record_count), '']));
                                    Progress.Log(StringOfChar('-', CHAR_LENGTH));

                                    Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                                    chatsync_message_found_count := 0;
                                    h_startpos := 0;
                                    HeaderReader.Position := 0;
                                    HeaderRegex.Stream := HeaderReader;
                                    FooterRegEx.Stream := FooterReader;
                                    HeaderRegex.Find(h_offset, h_count); // Find the first match, h_offset returned is relative to start pos
                                    while h_offset <> -1 do // h_offset returned as -1 means no hit
                                      try
                                        // Found a Regex Header ========================================
                                        if not Progress.isRunning then
                                        break;
                                        HeaderReader.Position := h_startpos + h_offset + 3;

                                        // Now look for Regex Footer -----------------------------------------
                                        FooterRegEx.Stream.Position := HeaderReader.Position;
                                        end_pos_max := HeaderReader.Position + 512; // Limit looking for the footer to our max size
                                        if end_pos_max >= HeaderRegex.Stream.Size then
                                        end_pos_max := HeaderRegex.Stream.Size - 1; // Don't go past the end!
                                        FooterRegEx.Stream.Size := end_pos_max;
                                        FooterRegEx.Find(f_offset, f_count);

                                        if (f_offset <> -1) and (f_offset + f_count >= 0) then // Found a footer and the size was at least our minimum
                                        begin
                                        // Found Footer ==============================================
                                        chatsync_beg_int64 := HeaderReader.Position;
                                        chatsync_end_int64 := chatsync_beg_int64 + f_offset + f_count - 1; // -1 is the length of the footer search term

                                        // Read the message as printable char ************************
                                        chatsync_msg_str := HeaderReader.AsPrintableChar(chatsync_end_int64 - HeaderReader.Position); // Header reader is now at the end of the message.
                                        if Trim(chatsync_msg_str) <> '' then
                                        begin

                                        // Walk back to the marker to work out the position of the record date/time
                                        walk_back_pos := HeaderReader.Position;
                                        for j := 1 to 10000 do
                                        begin
                                        walk_back_pos := walk_back_pos - 1;
                                        HeaderReader.Position := walk_back_pos;
                                        if HeaderReader.Position < 0 then
                                        break; // Do not go past the beginning of the file
                                        hexbytes_str := Trim(HeaderReader.AsHexString(4));
                                        if hexbytes_str = '41 01 05 02' then
                                        begin
                                        HeaderReader.Position := HeaderReader.Position - 16;
                                        break;
                                        end;
                                        end;

                                        chatsync_MsgDate_int64 := HeaderReader.AsInteger;
                                        if DateCheck_Unix_10(chatsync_MsgDate_int64) then
                                        begin
                                        chatsync_MsgDate_str := '';
                                        chatsync_MsgDate_dt := UnixTimeToDateTime(chatsync_MsgDate_int64);
                                        if (Trim(DateTimeToStr(chatsync_MsgDate_dt)) <> '') then
                                        chatsync_MsgDate_str := (DateTimeToStr(chatsync_MsgDate_dt));
                                        end;

                                        // Print the header only for first found record
                                        chatsync_message_found_count := chatsync_message_found_count + 1;
                                        if chatsync_message_found_count = 1 then
                                        begin
                                        Progress.Log(format(FORMAT_STR, ['', '#', 'Date/Time', 'Message Offset', 'Message']));
                                        Progress.Log(format(FORMAT_STR, ['', '-', '---------', '--------------', '-------']));
                                        end;

                                        // Print the record !!!
                                        Progress.Log(format(FORMAT_STR, ['', IntToStr(chatsync_message_found_count) + '.', chatsync_MsgDate_str, IntToStr(chatsync_beg_int64) + HYPHEN_NS + IntToStr(chatsync_end_int64), chatsync_msg_str]));

                                        try
                                        variant_Array[1] := chatsync_MsgDate_dt;
                                        except
                                        variant_Array[1] := 0;
                                        end;

                                        try
                                        variant_Array[2] := chatsync_users_str;
                                        except
                                        variant_Array[2] := '';
                                        end;

                                        try
                                        variant_Array[3] := chatsync_msg_str;
                                        except
                                        variant_Array[3] := '';
                                        end;

                                        AddToModule;

                                        end;
                                        end;

                                        HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match for the current file
                                      except
                                        Progress.Log(ATRY_EXCEPT_STR + RPad('!!!!!!!ERROR!!!!!!! - IN PROCESS LOOP:', RPAD_VALUE) + aArtifactEntry.EntryName);
                                      end;
                                  end;
                                except
                                  Progress.Log(ATRY_EXCEPT_STR + RPad('!!!!!!!ERROR!!!!!!! - COULD NOT OPEN:', RPAD_VALUE) + aArtifactEntry.EntryName);
                                end;
                              if chatsync_message_found_count = 0 then
                                Progress.Log(PAD + 'No messages.');
                              Progress.Log(StringOfChar('=', CHAR_LENGTH));
                            end;

                            // Process CHT_IOS_TELEGRAM_USER =====================
                            if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_TELEGRAM_USER') then
                            begin
                              // User ID -----------------------------------------
                              CarvedEntryReader.Position := 12;
                              if Trim(CarvedEntryReader.AsHexString(1)) = '69' then // noslz - Finds 'i' for Telegram ID
                              begin
                                CarvedEntryReader.Position := 14; // Gets the ID from position 14 + 4 bytes
                                CarvedEntryReader.Read(Telegram_UserID, 4);
                                variant_Array[1] := IntToStr(Telegram_UserID);
                              end;
                              // First Name --------------------------------------
                              Re.MatchPattern := '(?<=\.fn\.)(.*)(?=(\.ln\.|\.p\.|\.un\.))'; // noslz - regex carved_str
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := StringReplace(Re.matchedstr, '..', '', [rfReplaceAll]);
                                variant_Array[2] := Trim(temp_str);
                              end;
                              // Last Name ---------------------------------------
                              Re.MatchPattern := '(?<=\.ln\.)(.*)(?=\.p\.|\.un\.)'; // noslz - regex carved_str
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := StringReplace(Re.matchedstr, '..', '', [rfReplaceAll]);
                                variant_Array[3] := Trim(temp_str);
                              end;
                              // User Name ---------------------------------------
                              Re.MatchPattern := '(?<=\.un\.)(.*)(?=\.p\.|\.ph\.)'; // noslz - regex carved_str
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := StringReplace(Re.matchedstr, '..', '', [rfReplaceAll]);
                                variant_Array[4] := Trim(temp_str);
                              end;
                              // Phone Number ------------------------------------
                              Re.MatchPattern := '(?<=\.p\.)(.*)(?=\.ph\.)'; // noslz - regex carved_str
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := StringReplace(Re.matchedstr, '..', '', [rfReplaceAll]);
                                variant_Array[5] := Trim(temp_str);
                              end;
                              variant_Array[6] := 'carved sql'; // noslz
                            end;
                            if assigned(Re) then
                              FreeAndNil(Re);
                            AddToModule;

                            // Process Instagram =================================
                            if (UpperCase(Item^.fi_Process_ID) = 'CHT_IOS_INSTAGRAM_CONTACTS') then
                            begin
                              // ID ----------------------------------------------
                              Re.MatchPattern := '(?<=\.{29}[A-Z])(\d{5,10})(?=\xD3)'; // noslz
                              if Re.Match(0) >= 0 then
                                variant_Array[2] := Re.matchedstr;
                              // URL----------------------------------------------
                              Re.MatchPattern := 'https:(\/\/instagram).*(\d{8}_).*_[a-z]\.jpg';
                              if Re.Match(0) >= 0 then
                                variant_Array[3] := Re.matchedstr;
                              // NAME---------------------------------------------
                              temp_str := '';
                              Re.MatchPattern := '\d{19}.' + variant_Array[2] + '.*\.\.\.\.\.\.';
                              if Re.Match(0) >= 0 then
                              begin
                                temp_str := Re.matchedstr;
                                Delete(temp_str, 1, 19);
                                temp_str := StringReplace(temp_str, '_' + variant_Array[2], '', [rfReplaceAll]);
                                Delete(temp_str, 1, 1);
                                StripIllegalChars(temp_str);
                                temp_str := StringReplace(temp_str, '..', '', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '\', ' ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '[', ' ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, ']', ' ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '^', ' ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '#@', ' ', [rfReplaceAll]);
                                temp_str := StringReplace(temp_str, '_.J', ' ', [rfReplaceAll]);
                                if temp_str[1] = '_' then
                                  Delete(temp_str, 1, 1);
                                if (temp_str[Length(temp_str)]) = '_' then
                                  SetLength(temp_str, Length(temp_str) - 1);
                                Trim(temp_str);
                                variant_Array[1] := temp_str;
                              end;
                            end;
                            if assigned(Re) then
                              FreeAndNil(Re);
                            AddToModule;
                          end;
                        end;
                      end;
                      HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match
                    end;
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + aArtifactEntry.EntryName);
                  end; { Regex Carve }
              end;

              // ================================================================
              // YAHOO MESSENGER DAT
              // ================================================================
              // https://cryptome.org/isp-spy/yahoo-chat-spy.pdf
              // http://www.0xcafefeed.com/2007/12/yahoo-messenger-archive-file-format/

              if (UpperCase(Item^.fi_Process_ID) = 'CHT_PC_YAHOO_MESSENGER_WINDOWS') then
              begin
                newEntryReader.Position := 0;

                Progress.Log('Processing: ' + aArtifactEntry.EntryName);

                // Get the Archive Name from the folder ------------------------
                YahooName := '';
                if Length(aArtifactEntry.EntryName) > 13 then
                  YahooName := copy(aArtifactEntry.EntryName, 10, Length(aArtifactEntry.EntryName) - 13);

                while Progress.isRunning and (newEntryReader.Position < newEntryReader.Size) do
                begin
                  for g := 1 to ColCount do
                    variant_Array[g] := null; // Null the array

                  // Yahoo Date ------------------------------------------------
                  newEntryReader.Read(YahooDword, 4);
                  YahooDateTime := UnixTimeToDateTime(int64(YahooDword));
                  variant_Array[1] := YahooDateTime;

                  // Local User
                  variant_Array[2] := YahooName;

                  // Yahoo Type (06 = Searchable) ------------------------------
                  newEntryReader.Read(YahooType, 4);
                  if YahooType = 6 then
                    variant_Array[3] := 'Searchable'
                  else
                    variant_Array[3] := IntToStr(YahooType);

                  // Yahoo Sent/Received ---------------------------------------
                  newEntryReader.Read(YahooUser, 4);
                  if (YahooUser = 1) then
                    variant_Array[5] := 'Received'
                  else if (YahooUser = 0) then
                    variant_Array[5] := 'Sent'
                  else
                    variant_Array[5] := 'Unknown';

                  // Yahoo Message Length --------------------------------------
                  newEntryReader.Read(YahooMsgLength, 4);

                  // Decode Yahoo Message --------------------------------------
                  if (YahooMsgLength > 0) and (YahooMsgLength < 1000) then
                  begin
                    FillChar(YahooMsgBytes, 1000, 0);
                    newEntryReader.Read(YahooMsgBytes[1], YahooMsgLength);
                    if Length(YahooName) > 0 then
                    begin
                      cbyte := 1;
                      YahooDecodedMsg := '';
                      for yn := 1 to YahooMsgLength do
                      begin
                        cypherbyte := Ord(YahooName[cbyte]);
                        YahooMsgBytes[yn] := YahooMsgBytes[yn] xor cypherbyte;
                        Inc(cbyte);
                        if cbyte > Length(YahooName) then
                          cbyte := 1;
                        if YahooMsgBytes[yn] <> 0 then
                          YahooDecodedMsg := YahooDecodedMsg + chr(YahooMsgBytes[yn]);
                      end;
                      variant_Array[4] := (YahooDecodedMsg);
                    end;
                  end
                  else if YahooMsgLength <> 0 then
                    break;

                  // Yahoo Terminator Length -----------------------------------
                  newEntryReader.Read(YahooTermLength, 4);

                  // Get Conference User ---------------------------------------
                  YahooConfUser_str := '';
                  if (YahooTermLength > 0) and (YahooTermLength <= 255) then
                  begin
                    FillChar(YahooConfUserBytes[1], 1000, 0);
                    newEntryReader.Read(YahooConfUserBytes[1], YahooTermLength);
                    for g := 1 to YahooTermLength do
                      YahooConfUser_str := YahooConfUser_str + chr(YahooConfUserBytes[g]);
                  end;

                  // Other User ------------------------------------------------
                  if YahooConfUser_str <> '' then
                    variant_Array[6] := YahooConfUser_str
                  else
                    variant_Array[6] := (aArtifactEntry.Parent.EntryName);

                  // Archive Folder --------------------------------------------
                  variant_Array[7] := aArtifactEntry.Parent.EntryName;

                  // Archive Type ----------------------------------------------
                  variant_Array[8] := aArtifactEntry.Parent.Parent.EntryName;

                  AddToModule;
                end; { while }
              end; { Yahoo Messenger.dat }

              //Progress.Log(StringOfChar('D', CHAR_LENGTH));
            end;
            Progress.IncCurrentprogress;
          end;

          // Add to the gArtifactsDataStore
          if assigned(ADDList) and Progress.isRunning then
          begin
            if (CmdLine.Params.Indexof(TRIAGE) = -1) then
              gArtifactsDataStore.Add(ADDList);
            Progress.Log(RPad('Total Artifacts:', RPAD_VALUE) + IntToStr(ADDList.Count));

            // Export L01 files where artifacts are found (controlled by Artifact_Utils.pas)
            if (gArr_ValidatedFiles_TList[ref_num].Count > 0) and (ADDList.Count > 0) then
              ExportToL01(gArr_ValidatedFiles_TList[ref_num], Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type);

            ADDList.Clear;
          end;

          Progress.Log(StringOfChar('-', CHAR_LENGTH));
        finally
          records_read_int := 0;
          FreeAndNil(CarvedEntryReader);
          FreeAndNil(FooterProgress);
          FreeAndNil(FooterReader);
          FreeAndNil(FooterRegEx);
          FreeAndNil(HeaderReader);
          FreeAndNil(HeaderRegex);
          FreeAndNil(ADDList);
          FreeAndNil(newEntryReader);
        end;
      end;

    end;

  end
  else
  begin
    Progress.Log(RPad('', RPAD_VALUE) + 'No files to process.');
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
  end;

end;

// ==========================================================================================================================================================
// Start of Script
// ==========================================================================================================================================================
const
  CREATE_GLOB_SEARCH_BL = False;
  SEARCHING_FOR_ARTIFACT_FILES_STR = 'Searching for Artifact files';

var
  AboutToProcess_StringList: TStringList;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  FolderEntry: TEntry;
  AllFoundList_count: integer;
  AllFoundListUnique: TUniqueListOfEntries;
  DeleteFolder_display_str: string;
  DeleteFolder_TList: TList;
  Enum: TEntryEnumerator;
  ExistingFolders_TList: TList;
  FieldItunesDomain: TDataStoreField;
  FieldItunesName: TDataStoreField;
  FindEntries_StringList: TStringList;
  FoundList, AllFoundList: TList;
  GlobSearch_StringList: TStringList;
  i: integer;
  Item: TSQL_FileSearch;
  iTunes_Domain_str: string;
  iTunes_Name_str: string;
  Process_ID_str: string;
  ResultInt: integer;

{$IF DEFINED (ISFEXGUI)}
  Display_StringList: TStringList;
  MyScriptForm: TScriptForm;
{$IFEND}

procedure LogExistingFolders(aExistingFolders_TList: TList; aDeleteFolder_TList: TList); // Compare About to Process Folders with Existing Artifact Module Folders
const
  LOG_BL = False;
var
  aFolderEntry: TEntry;
  idx, t: integer;
begin
  if LOG_BL then Progress.Log('Existing Folders:');
  for idx := 0 to aExistingFolders_TList.Count - 1 do
  begin
    if not Progress.isRunning then break;
    aFolderEntry := (TEntry(aExistingFolders_TList[idx]));
    if LOG_BL then
      Progress.Log(HYPHEN + aFolderEntry.FullPathName);
    for t := 0 to AboutToProcess_StringList.Count - 1 do
    begin
      if aFolderEntry.FullPathName = AboutToProcess_StringList[t] then
        aDeleteFolder_TList.Add(aFolderEntry);
    end;
  end;
  if LOG_BL then
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

// Start of script -------------------------------------------------------------
var
  anEntry: TEntry;
  anint: integer;
  param_str: string;
  progress_program_str: string;
  ref_num: integer;
  regex_search_str: string;
  n, r, s: integer;
  temp_int: integer;
  test_param_int: integer;
  temp_process_counter: integer;

begin
  Read_SQLite_DB;

  SetLength(gArr_ValidatedFiles_TList, gNumberOfSearchItems);
  SetLength(gArtConnect_ProgFldr, gNumberOfSearchItems);

  Progress.Log(SCRIPT_NAME + ' started' + RUNNING);
  if (CmdLine.Params.Indexof(TRIAGE) > -1) then
    Progress.DisplayTitle := 'Triage' + HYPHEN + 'File System' + HYPHEN + CATEGORY_NAME
  else
    Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME;
  Progress.LogType := ltVerbose; // ltOff, ltVerbose, ltDebug, ltTechnical

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := ('Processing complete.');
    Exit;
  end;

  // Parameters Received -------------------------------------------------------
  param_str := '';
  test_param_int := 0;
  gParameter_Num_StringList := TStringList.Create;
  try
//    if CmdLine.ParamCount = 0 then
//    begin
//      MessageUser('No processing parameters received.');
//      Exit;
//    end
//    else
    begin
      Progress.Log(RPad('Parameters Received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
      for n := 0 to CmdLine.ParamCount - 1 do
      begin
        if not Progress.isRunning then
          break;

        param_str := param_str + '"' + CmdLine.Params[n] + '"' + ' ';
        // Validate Parameters
        if (RegexMatch(CmdLine.Params[n], '\d{1,2}$', False)) then
        begin
          try
            test_param_int := StrToInt(CmdLine.Params[n]);
            if (test_param_int <= -1) or (test_param_int > Length(gArr) - 1) then
            begin
              MessageUser('Invalid parameter received: ' + (CmdLine.Params[n]) + CR + ('Maximum is: ' + IntToStr(Length(gArr) - 1) + DCR + SCRIPT_NAME + ' will terminate.'));
              Exit;
            end;
            gParameter_Num_StringList.Add(CmdLine.Params[n]);
            Item := gArr[test_param_int];
            Progress.Log(RPad(HYPHEN + 'param_str ' + IntToStr(n) + ' = ' + CmdLine.Params[n], RPAD_VALUE) + format('%-10s %-10s %-25s %-12s', ['Ref#: ' + CmdLine.Params[n], CATEGORY_NAME,
              Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, Item.fi_Name_OS]));
          except
            begin
              Progress.Log(ATRY_EXCEPT_STR + 'Error validating parameter. ' + SCRIPT_NAME + ' will terminate.');
              Exit;
            end;
          end;
        end;
      end;
      Trim(param_str);
    end;
    Progress.Log(RPad('param_str:', RPAD_VALUE) + param_str); // noslz
    Progress.Log(StringOfChar('-', CHAR_LENGTH));

    // Progress Bar Text
    progress_program_str := '';
    if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      progress_program_str := PROGRAM_NAME;
    end;

    // Create GlobSearch -------------------------------------------------------
    if CREATE_GLOB_SEARCH_BL then
    begin
      GlobSearch_StringList := TStringList.Create;
      GlobSearch_StringList.Sorted := True;
      GlobSearch_StringList.Duplicates := dupIgnore;
      try
        for n := 0 to Length(gArr) - 1 do
          Create_Global_Search(n, gArr[n], GlobSearch_StringList);
        Progress.Log(GlobSearch_StringList.Text);
      finally
        GlobSearch_StringList.free;
      end;
      Exit;
    end;

    for n := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      if (CmdLine.Params.Indexof(IntToStr(n)) > -1) then
      begin
        Item := gArr[n];
        progress_program_str := 'Ref#:' + SPACE + param_str + SPACE + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type;
        break;
      end;
    end;

    if progress_program_str = '' then
      progress_program_str := 'Other';

    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;

    try
      gFileSystemDataStore := GetDataStore(DATASTORE_FILESYSTEM);
      if not assigned(gFileSystemDataStore) then
      begin
        Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + DCR + TSWT);
        Exit;
      end;

      try
        FieldItunesDomain := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_DOMAIN);
        FieldItunesName := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_NAME);

        // Create TLists For Valid Files
        for n := 0 to Length(gArr) - 1 do
        begin
          if not Progress.isRunning then
            break;
          gArr_ValidatedFiles_TList[n] := TList.Create;
        end;

        try
          AboutToProcess_StringList := TStringList.Create;
          try
            if (CmdLine.Params.Indexof('MASTER') = -1) then
            begin
              if (CmdLine.Params.Indexof('NOSHOW') = -1) then
              begin
                // PROCESSALL - Create the AboutToProcess_StringList
                if (CmdLine.Params.Indexof(PROCESSALL) > -1) or (CmdLine.ParamCount = 0) then
                begin
                  for n := 0 to Length(gArr) - 1 do
                  begin
                    if not Progress.isRunning then break;
                    Item := gArr[n];
                    AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[n]));
                    Progress.Log(format('%-4s %-59s %-30s', ['#' + IntToStr(n), Item.fi_Process_ID, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type]));
                  end;
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end
                else

                // PROCESS INDIVIDUAL - Create the AboutToProcess_StringList
                begin
                  if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
                  begin
                    for n := 0 to gParameter_Num_StringList.Count - 1 do
                    begin
                      if not Progress.isRunning then break;
                      temp_int := StrToInt(gParameter_Num_StringList[n]);
                      Item := gArr[temp_int];
                      AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[temp_int]));
                    end;
                  end;
                end;

                if CmdLine.ParamCount = 0 then
                begin
                  Progress.Log('No parameters received. Use a number or "PROCESSALL".');
                  Exit;
                end;

                // Show the form
                ResultInt := 1; // Continue AboutToProcess

{$IF DEFINED (ISFEXGUI)}
                Display_StringList := TStringList.Create;
                Display_StringList.Sorted := True;
                Display_StringList.Duplicates := dupIgnore;
                try
                  for i := 0 to AboutToProcess_StringList.Count - 1 do
                  begin
                    Display_StringList.Add(ExtractFileName(AboutToProcess_StringList[i]));
                  end;

                  if assigned(AboutToProcess_StringList) and (AboutToProcess_StringList.Count > 30) then
                  begin
                    MyScriptForm := TScriptForm.Create;
                    try
                      MyScriptForm.SetCaption(SCRIPT_DESCRIPTION);
                      MyScriptForm.SetText(Display_StringList.Text);
                      ResultInt := idCancel;
                      if MyScriptForm.ShowModal then
                        ResultInt := idOk
                    finally
                      FreeAndNil(MyScriptForm);
                    end;
                  end
                  else
                  begin
                    ResultInt := MessageBox('Extract Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME, (MB_OKCANCEL or MB_ICONQUESTION or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                  end;
                finally
                  Display_StringList.free;
                end;
{$IFEND}
              end;

              if ResultInt > 1 then
              begin
                Progress.Log(CANCELED_BY_USER);
                Progress.DisplayMessageNow := CANCELED_BY_USER;
                Exit;
              end;

            end;

            // Deal with Existing Artifact Folders
            ExistingFolders_TList := TList.Create;
            DeleteFolder_TList := TList.Create;
            try
              if gArtifactsDataStore.Count > 1 then
              begin
                anEntry := gArtifactsDataStore.First;
                while assigned(anEntry) and Progress.isRunning do
                begin
                  if anEntry.isDirectory then
                    ExistingFolders_TList.Add(anEntry);
                  anEntry := gArtifactsDataStore.Next;
                end;
                gArtifactsDataStore.Close;
              end;

              LogExistingFolders(ExistingFolders_TList, DeleteFolder_TList);

              // Create the delete folder TList and display string
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                Progress.Log('Replacing folders:');
                for s := 0 to DeleteFolder_TList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  FolderEntry := TEntry(DeleteFolder_TList[s]);
                  DeleteFolder_display_str := DeleteFolder_display_str + #13#10 + FolderEntry.FullPathName;
                  Progress.Log(HYPHEN + FolderEntry.FullPathName);
                end;
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;

              // Message Box - When artifact folder is already present ---------------------
{$IF DEFINED (ISFEXGUI)}
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                if (CmdLine.Params.Indexof('MASTER') = -1) and (CmdLine.Params.Indexof('NOSHOW') = -1) then
                begin

                  if (CmdLine.Params.Indexof('NOSHOW') = -1) then
                  begin
                    if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
                    begin
                      ResultInt := MessageBox('Artifacts have already been processed:' + #13#10 + DeleteFolder_display_str + DCR + 'Replace the existing Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME,
                        (MB_OKCANCEL or MB_ICONWARNING or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                    end;
                  end
                  else
                    ResultInt := idOk;
                  case ResultInt of
                    idOk:
                      begin
                        try
                          gArtifactsDataStore.Remove(DeleteFolder_TList);
                        except
                          MessageBox('ERROR: There was an error deleting existing artifacts.' + #13#10 + 'Save then reopen your case.', SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
                        end;
                      end;
                    idCancel:
                      begin
                        Progress.Log(CANCELED_BY_USER);
                        Progress.DisplayMessageNow := CANCELED_BY_USER;
                        Exit;
                      end;
                  end;
                end;
              end;
{$IFEND}

            finally
              ExistingFolders_TList.free;
              DeleteFolder_TList.free;
            end;
          finally
            AboutToProcess_StringList.free;
          end;

          // Find iTunes Backups and run Signature Analysis
          if (CmdLine.Params.Indexof('NOITUNES') = -1) then
            Itunes_Backup_Signature_Analysis;

          // Create the RegEx Search String
          regex_search_str := '';
          begin
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if (CmdLine.Params.Indexof(IntToStr(n)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                if Item.fi_Regex_Search <> ''      then regex_search_str := regex_search_str + '|' + Item.fi_Regex_Search;
                if Item.fi_Rgx_Itun_Bkup_Dmn <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Dmn;
                if Item.fi_Rgx_Itun_Bkup_Nme <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Nme;
              end;
            end;
          end;
          if (regex_search_str <> '') and (regex_search_str[1] = '|') then
            Delete(regex_search_str, 1, 1);

          AllFoundList := TList.Create;
          try
            AllFoundListUnique := TUniqueListOfEntries.Create;
            FoundList := TList.Create;
            FindEntries_StringList := TStringList.Create;
            FindEntries_StringList.Sorted := True;
            FindEntries_StringList.Duplicates := dupIgnore;
            try
              if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                Progress.Log('Find files by path (PROCESSALL)' + RUNNING);
                for i := low(gArr) to high(gArr) do
                begin
                  if (gArr[i].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob1_Search);
                  if (gArr[i].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob2_Search);
                  if (gArr[i].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob3_Search);
                end;
              end
              else
              begin
                Progress.Log('Find files by path (Individual)' + RUNNING);
                for i := 0 to gParameter_Num_StringList.Count - 1 do
                begin
                  anint := StrToInt(gParameter_Num_StringList[i]);
                  if (gArr[anint].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob1_Search);
                  if (gArr[anint].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob2_Search);
                  if (gArr[anint].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob3_Search);
                end;
              end;

              // Find the files by path and add to AllFoundListUnique
              Progress.Initialize(FindEntries_StringList.Count, STR_FILES_BY_PATH + RUNNING);
              gtick_foundlist_i64 := GetTickCount;
              for i := 0 to FindEntries_StringList.Count - 1 do
              begin
                if not Progress.isRunning then
                  break;
                try
                  Find_Entries_By_Path(gFileSystemDataStore, FindEntries_StringList[i], FoundList, AllFoundListUnique);
                except
                  Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Find_Entries_By_Path');
                end;
                Progress.IncCurrentprogress;
              end;

              Progress.Log(RPad(STR_FILES_BY_PATH + SPACE + '(Unique)' + COLON, RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Add > 40 char files with no extension to Unique List (possible iTunes backup files)
              Add40CharFiles(AllFoundListUnique);
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Move the AllFoundListUnique list into a TList
              if assigned(AllFoundListUnique) and (AllFoundListUnique.Count > 0) then
              begin
                Enum := AllFoundListUnique.GetEnumerator;
                while Enum.MoveNext do
                begin
                  anEntry := Enum.Current;
                  AllFoundList.Add(anEntry);
                end;
              end;

              // Now work with the TList from now on
              if assigned(AllFoundList) and (AllFoundList.Count > 0) then
              begin
                Progress.Log(RPad('Unique Files by path:', RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
                Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Add flags
                if BL_USE_FLAGS then
                begin
                  Progress.Log('Adding flags' + RUNNING);
                  for i := 0 to AllFoundList.Count - 1 do
                  begin
                    if not Progress.isRunning then
                      break;
                    anEntry := TEntry(AllFoundList[i]);
                    anEntry.Flags := anEntry.Flags + [Flag7];
                  end;
                  Progress.Log('Finished adding flags' + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Determine signature
                if assigned(AllFoundList) and (AllFoundList.Count > 0) then
                  SignatureAnalysis(AllFoundList);

                gtick_foundlist_str := (RPad(STR_FILES_BY_PATH + COLON, RPAD_VALUE) + CalcTimeTaken(gtick_foundlist_i64));
              end;

            finally
              FoundList.free;
              AllFoundListUnique.free;
              FindEntries_StringList.free;
            end;

            if assigned(AllFoundList) and (AllFoundList.Count > 0) then
            begin
              Progress.Log(SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              Progress.Log(format(GFORMAT_STR, ['', 'Action', 'Ref#', 'Bates', 'Signature', 'Filename (trunc)', 'Reason'])); // noslz
              Progress.Log(format(GFORMAT_STR, ['', '------', '----', '-----', '---------', '----------------', '------'])); // noslz

              AllFoundList_count := AllFoundList.Count;
              Progress.Initialize(AllFoundList_count, SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              for i := 0 to AllFoundList.Count - 1 do
              begin
                Progress.DisplayMessages := SEARCHING_FOR_ARTIFACT_FILES_STR + SPACE + '(' + IntToStr(i) + ' of ' + IntToStr(AllFoundList_count) + ')' + RUNNING;
                if not Progress.isRunning then
                  break;
                anEntry := TEntry(AllFoundList[i]);

                if (i mod 10000 = 0) and (i > 0) then
                begin
                  Progress.Log('Processing: ' + IntToStr(i) + ' of ' + IntToStr(AllFoundList.Count) + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Set the iTunes Domain String
                iTunes_Domain_str := '';
                if assigned(FieldItunesDomain) then
                begin
                  try
                    iTunes_Domain_str := FieldItunesDomain.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Domain string');
                  end;
                end;

                // Set the iTunes Name String
                iTunes_Name_str := '';
                if assigned(FieldItunesName) then
                begin
                  try
                    iTunes_Name_str := FieldItunesName.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Name string');
                  end;
                end;

                // Run the match
                aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
                if
                (anEntry.Extension <> 'db-shm') and
                (anEntry.Extension <> 'db-wal') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite WAL') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite SHM') then
                begin
                  if RegexMatch(anEntry.EntryName, regex_search_str, False) or
                  RegexMatch(anEntry.FullPathName, regex_search_str, False) or
                  FileSubSignatureMatch(anEntry) or
                  RegexMatch(iTunes_Domain_str, regex_search_str, False) or
                  RegexMatch(iTunes_Name_str, regex_search_str, False) then
                  begin

                    // Sub Signature Match
                    if FileSubSignatureMatch(anEntry) then
                    begin
                      if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag2]; // Blue Flag
                      DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                    end
                    else

                    // Regex Name Match
                    begin
                      if ((not anEntry.isDirectory) or (anEntry.isDevice)) and (anEntry.LogicalSize > 0) and (anEntry.PhysicalSize > 0) then
                      begin
                        if FileNameRegexSearch(anEntry, iTunes_Domain_str, iTunes_Name_str, regex_search_str) then
                        begin
                          if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag1]; // Red Flag
                          DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                        end;
                      end;
                    end;

                  end;
                  Progress.IncCurrentprogress;
                end;
              end;
            end;
          finally
            AllFoundList.free;
          end;

          // Check to see if files were found
          if (TotalValidatedFileCountInTLists = 0) then
          begin
            Progress.Log('No ' + PROGRAM_NAME + SPACE + 'files were found.');
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Not found';
          end
          else
          begin
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
            Progress.Log(RPad('Total Validated Files:', RPAD_VALUE) + IntToStr(TotalValidatedFileCountInTLists));
            Progress.Log(StringOfChar('=', CHAR_LENGTH + 80));
          end;

          // Display the content of the TLists for further processing
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.Log('Lists available to process' + RUNNING);
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if gArr_ValidatedFiles_TList[n].Count > 0 then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log(RPad('TList ' + IntToStr(n) + ': ' + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type + HYPHEN + Item.fi_Name_OS, RPAD_VALUE) + IntToStr(gArr_ValidatedFiles_TList[n].Count));
                for r := 0 to gArr_ValidatedFiles_TList[n].Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  anEntry := (TEntry(TList(gArr_ValidatedFiles_TList[n]).items[r]));
                  if not(Item.fi_Process_As = 'POSTPROCESS') then
                    Progress.Log(RPad(' Bates: ' + IntToStr(anEntry.ID), RPAD_VALUE) + anEntry.EntryName + SPACE + Item.fi_Process_As);
                end;
              end;
            end;
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
          end;

          // *** CREATE GUI ***
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Found';

            Progress.Log(RPad('Process All Lists' + RUNNING, RPAD_VALUE) + IntToStr(High(gArr) + 1)); // noslz
            Progress.Log(StringOfChar('=', CHAR_LENGTH));

            // *** DO PROCESS ***
            temp_process_counter := 0;
            Progress.CurrentPosition := 0;
            Progress.Max := TotalValidatedFileCountInTLists;
            gtick_doprocess_i64 := GetTickCount;

            // *****************************************************************
            for n := 0 to Length(gArr) - 1 do
            begin
              ref_num := n;
              if (TestForDoProcess(ref_num)) then
              begin
                Item := gArr[n];
                Process_ID_str := Item.fi_Process_ID;
                DoProcess(gArtConnect_ProgFldr[ref_num], ref_num, Chat_Columns.GetTable(Process_ID_str));
              end;
            end;
            // *****************************************************************

            gtick_doprocess_str := (RPad('DoProcess:', RPAD_VALUE) + CalcTimeTaken(gtick_doprocess_i64));

          end;

          if not Progress.isRunning then
          begin
            Progress.Log(CANCELED_BY_USER);
            Progress.DisplayMessageNow := CANCELED_BY_USER;
          end;

        finally
          for n := 0 to Length(gArr) - 1 do
            FreeAndNil(gArr_ValidatedFiles_TList[n]);
        end;

      finally
        if assigned(gFileSystemDataStore) then
          FreeAndNil(gFileSystemDataStore);
      end;

    finally
      if assigned(gArtifactsDataStore) then
        FreeAndNil(gArtifactsDataStore);
    end;

  finally
    if assigned(gParameter_Num_StringList) then
      FreeAndNil(gParameter_Num_StringList);
  end;

  if Progress.isRunning then
  begin
    Progress.Log(gtick_foundlist_str);
    Progress.Log(gtick_doprocess_str);
  end;

  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  Progress.Log(SCRIPT_NAME + ' finished.');
  Progress.DisplayMessageNow := ('Processing complete.');

end.
