﻿unit Artifacts_Locations;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  ByteStream, Classes, Clipbrd, Common, Contnrs, DataEntry, DataStorage, DateUtils, DIRegex,
  Graphics, JSON, PropertyList, Math, ProtoBuf, Regex, SQLite3, SysUtils, Variants, XML,
  Artifact_Utils      in 'Common\Artifact_Utils.pas',
  Columns             in 'Common\Column_Arrays\Columns.pas',
  Icon_List           in 'Common\Icon_List.pas',
  Itunes_Backup_Sig   in 'Common\Itunes_Backup_Signature_Analysis.pas',
  RS_DoNotTranslate   in 'Common\RS_DoNotTranslate.pas',
  Location_Columns    in 'Common\Column_Arrays\Arrays_Location.pas';

type
  TSearchResult = record
    Address: string;
    Latitude: Double;
    Longitude: Double;
  end;

type
  TDataStoreFieldArray = array of TDataStoreField;

  TSQL_FileSearch = record
    fi_Carve_Adjustment           : integer;
    fi_Carve_Footer               : string;
    fi_Carve_Header               : string;
    fi_Glob1_Search               : string;
    fi_Glob2_Search               : string;
    fi_Glob3_Search               : string;
    fi_Icon_Category              : integer;
    fi_Icon_OS                    : integer;
    fi_Icon_Program               : integer;
    fi_Name_OS                    : string;
    fi_Name_Program               : string;
    fi_Name_Program_Type          : string;
    fi_NodeByName                 : string;
    fi_Process_As                 : string;
    fi_Process_ID                 : string;
    fi_Reference_Info             : string;
    fi_Regex_Search               : string;
    fi_Rgx_Itun_Bkup_Dmn          : string;
    fi_Rgx_Itun_Bkup_Nme          : string;
    fi_RootNodeName               : string;
    fi_Signature_Parent           : string;
    fi_Signature_Sub              : string;
    fi_SQLPrimary_Tablestr        : string;
    fi_SQLStatement               : string;
    fi_SQLTables_Required         : string;
    fi_Test_Data                  : string;
  end;

  PSQL_FileSearch = ^TSQL_FileSearch;
  TgArr = array of TSQL_FileSearch;

const
  ARTIFACTS_DB              = 'Artifacts_Locations.db';
  CATEGORY_NAME             = 'Locations';
  PROGRAM_NAME              = 'Location';
  SCRIPT_DESCRIPTION        = 'Extract Location Artifacts';
  SCRIPT_NAME               = 'Location.pas';
  // ~~~
  ATRY_EXCEPT_STR           = 'TryExcept: '; // noslz
  ANDROID                   = 'Android'; // noslz
  BL_BOOKMARK_SOURCE        = False;
  BL_PROCEED_LOGGING        = False;
  BL_USE_FLAGS              = False;
  BS                        = '\';
  CANCELED_BY_USER          = 'Canceled by user.';
  CHAR_LENGTH               = 80;
  COLON                     = ':';
  COMMA                     = ',';
  CR                        = #13#10;
  DCR                       = #13#10 + #13#10;
  DUP_FILE                  = '( \d*)?$';
  FBN_ITUNES_BACKUP_DOMAIN  = 'iTunes Backup Domain'; // noslz
  FBN_ITUNES_BACKUP_NAME    = 'iTunes Backup Name'; // noslz
  GFORMAT_STR               = '%-1s %-12s %-8s %-15s %-25s %-30s %-20s';
  GROUP_NAME                = 'Artifact Analysis';
  HYPHEN                    = ' - ';
  HYPHEN_NS                 = '-';
  ICON_ANDROID              = 1017;
  ICON_IOS                  = 1062;
  IOS                       = 'iOS'; // noslz
  MAX_CARVE_SIZE            = 1024 * 10;
  MIN_CARVE_SIZE            = 0;
  OSFILENAME                = '\\?\'; // This is needed for long filenames
  PIPE                      = '|'; // noslz
  PROCESS_AS_BYTES_STRLST   = 'PROCESS_AS_BYTES_STRLST'; // noslz
  PROCESS_AS_CARVE          = 'PROCESSASCARVE'; // noslz
  PROCESS_AS_PLIST          = 'PROCESSASPLIST'; // noslz
  PROCESS_AS_PYTHON         = 'PROCESSASPYTHON'; // noslz
  PROCESS_AS_TEXT           = 'PROCESSASTEXT'; // noslz
  PROCESS_AS_XML            = 'PROCESSASXML'; // noslz
  PROCESSALL                = 'PROCESSALL'; // noslz
  RPAD_VALUE                = 55;
  RUNNING                   = '...';
  SPACE                     = ' ';
  STR_FILES_BY_PATH         = 'Files by path';
  TRIAGE                    = 'TRIAGE'; // noslz
  TSWT                      = 'The script will terminate.';
  VERBOSE                   = False;
  WINDOWS                   = 'Windows'; // noslz

var
  gArr:                           TgArr;
  gArr_ValidatedFiles_TList:      array of TList;
  gArtConnect_CatFldr:            TArtifactEntry1;
  gArtConnect_ProgFldr:           array of TArtifactConnectEntry;
  gArtifacts_SQLite:              TSQLite3Database;
  gArtifactsDataStore:            TDataStore;
  gdb_read_bl:                    boolean;
  gFileSystemDataStore:           TDataStore;
  gNumberOfSearchItems:           integer;
  gParameter_Num_StringList:      TStringList;
  gportablepython_path_str:       string;
  gPythonDB_pth:                  string;
  gPython_SQLite:                 TSQLite3Database;
  gtick_doprocess_i64:            uint64;
  gtick_doprocess_str:            string;
  gtick_foundlist_i64:            uint64;
  gtick_foundlist_str:            string;

function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
function FileSubSignatureMatch(Entry: TEntry): boolean;
function GetFullName(Item: PSQL_FileSearch): string;
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
function RPad(const AString: string; AChars: integer): string;
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
function TestForDoProcess(ARefNum: integer): boolean;
function TotalValidatedFileCountInTLists: integer;
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);

{$IF DEFINED (ISFEXGUI)}
type
  TScriptForm = class(TObject)
  private
    frmMain: TGUIForm;
    FMemo: TGUIMemoBox;
    pnlBottom: TGUIPanel;
    btnOK: TGUIButton;
    btnCancel: TGUIButton;
  public
    ModalResult: boolean;
    constructor Create;
    function ShowModal: boolean;
    procedure OKClick(Sender: TGUIControl);
    procedure CancelClick(Sender: TGUIControl);
    procedure SetText(const Value: string);
    procedure SetCaption(const Value: string);
  end;
{$IFEND}

implementation

{$IF DEFINED (ISFEXGUI)}

constructor TScriptForm.Create;
begin
  inherited Create;
  frmMain := NewForm(nil, SCRIPT_NAME);
  frmMain.Size(500, 480);
  pnlBottom := NewPanel(frmMain, esNone);
  pnlBottom.Size(frmMain.Width, 40);
  pnlBottom.Align(caBottom);
  btnOK := NewButton(pnlBottom, 'OK');
  btnOK.Position(pnlBottom.Width - 170, 4);
  btnOK.Size(75, 25);
  btnOK.Anchor(False, True, True, True);
  btnOK.DefaultBtn := True;
  btnOK.OnClick := OKClick;
  btnCancel := NewButton(pnlBottom, 'Cancel');
  btnCancel.Position(pnlBottom.Width - 88, 4);
  btnCancel.Size(75, 25);
  btnCancel.Anchor(False, True, True, True);
  btnCancel.CancelBtn := True;
  btnCancel.OnClick := CancelClick;
  FMemo := NewMemoBox(frmMain, [eoReadOnly]); // eoNoHScroll, eoNoVScroll
  FMemo.Color := clWhite;
  FMemo.Align(caClient);
  FMemo.FontName('Courier New');
  frmMain.CenterOnParent;
  frmMain.Invalidate;
end;

procedure TScriptForm.OKClick(Sender: TGUIControl);
begin
  // Set variables for use in the main proc. Must do this before closing the main form.
  ModalResult := False;
  ModalResult := True;
  frmMain.Close;
end;

procedure TScriptForm.CancelClick(Sender: TGUIControl);
begin
  ModalResult := False;
  Progress.DisplayMessageNow := CANCELED_BY_USER;
  frmMain.Close;
end;

function TScriptForm.ShowModal: boolean;
begin
  Execute(frmMain);
  Result := ModalResult;
end;

procedure TScriptForm.SetText(const Value: string);
begin
  FMemo.Text := Value;
end;

procedure TScriptForm.SetCaption(const Value: string);
begin
  frmMain.Text := Value;
end;
{$IFEND}

// -----------------------------------------------------------------------------
// Add 40 Character Files
// -----------------------------------------------------------------------------
procedure Add40CharFiles(UniqueList: TUniqueListOfEntries);
var
  Entry: TEntry;
  FileSystemDS: TDataStore;
begin
  FileSystemDS := GetDataStore(DATASTORE_FILESYSTEM);
  if assigned(UniqueList) and assigned(FileSystemDS) then
  begin
    Entry := FileSystemDS.First;
    while assigned(Entry) and Progress.isRunning do
    begin
      begin
        if RegexMatch(Entry.EntryName, '^[0-9a-fA-F]{40}', False) and (Entry.EntryNameExt = '') then // noslz iTunes Backup files
        begin
          UniqueList.Add(Entry);
        end;
      end;
      Entry := FileSystemDS.Next;
    end;
    FreeAndNil(FileSystemDS);
  end;
end;

function clean_str(aStr: string): string;
begin
  Result := aStr;
  Result := StringReplace(Result, '","', ', ', [rfReplaceAll]);
  Result := StringReplace(Result, '"', '', [rfReplaceAll]);
  Result := StringReplace(Result, '  ', ' ', [rfReplaceAll]);
end;

// -----------------------------------------------------------------------------
// Apple Maps Protobuf - ZROUTEREQUESTSTORAGE = "Search Data"
// -----------------------------------------------------------------------------
// Contains the original search query
// Stores the user's input parameters
// Includes the search context (e.g., starting from "194 St Paul St")
// Represents what the user was looking for
function Apple_Maps_ProtoBuf_Search_Data(bytes: TBytes; const type_str: string; record_int: integer) : TSearchResult;
const
  function_log_bl = False;
var
  pbr: TGDProtoBufReader;
  destStr: string;
  First1, Third1, Field: Cellptr;
  latRawInt, lonRawInt: Int64;
begin
  // Initialize Result with default values
  Result.Address := '';
  Result.Latitude := 0;
  Result.Longitude := 0;

  pbr := TGDProtoBufReader.Create;
  try
    pbr.LoadFromBytes(bytes);

    if type_str = 'ZROUTEREQUESTSTORAGE' then
    begin
      // Step 1: Read destination address
      destStr := pbr.ProBufLookupValue(pbr.rootcell, [1.2, 1, 2], [5, 6, 6.2, 6.3]);
      destStr := clean_str(destStr);
      if function_log_bl then Progress.Log('destStr: ' + destStr);
      Result.Address := destStr;

      // Step 2: root[1]
      First1 := pbr.ProBufLookupField(pbr.rootcell, [1]);
      if assigned(First1) then
      begin
        // Step 3: First1[3]
        Third1 := pbr.ProBufLookupField(First1, [3]);
        if assigned(Third1) then
        begin
          // Step 4: Read Latitude
          Field := pbr.ProBufLookupField(Third1, [1]);
          if assigned(Field) then
          begin
            latRawInt := pbr.GetCellInt64(Field);  // <-- FIX: read to variable first
            Move(latRawInt, Result.Latitude, SizeOf(Result.Latitude));
            if function_log_bl then Progress.Log('Func_Lat: ' + FloatToStr(Result.Latitude)); // noslz
          end
          else
            Progress.Log('Latitude field not found');
          // Step 5: Read Longitude
          Field := pbr.ProBufLookupField(Third1, [2]);
          if assigned(Field) then
          begin
            lonRawInt := pbr.GetCellInt64(Field);  // <-- FIX: read to variable first
            Move(lonRawInt, Result.Longitude, SizeOf(Result.Longitude));
            if function_log_bl then Progress.Log('Func_Lone: ' + FloatToStr(Result.Longitude)); // noslz
          end
          else
            if function_log_bl then Progress.Log('Longitude field not found');
        end
        else
          if function_log_bl then Progress.Log('Field 3 not found'); // noslz
      end
      else
        if function_log_bl then Progress.Log('Field 1 not found'); // noslz
    end;
  finally
    pbr.free;
  end;
end;

// -----------------------------------------------------------------------------
// Apple Maps Protobuf - ZMAPITEMSTORAGE = "Trip Data"
// -----------------------------------------------------------------------------
// Contains the destination details
// Stores precise coordinates needed for navigation
// Includes place-specific metadata for the trip destination
// Represents where the user was going

function Apple_Maps_ProtoBuf_Trip_Data(bytes: TBytes; const type_str: string) : string;
var
  array_of_cells: TArrayCells;
  array_of_cells_8: TArrayCells;
  cp: Cellptr;
  cp2: Cellptr;
  cp8: Cellptr;
  i: integer;
  k: integer;
  pbr: TGDProtoBufReader;
  str: string;
begin
  Result := '';
  pbr := TGDProtoBufReader.Create;
  try
    pbr.Loadfrombytes(bytes);

    // Get the Destination Address
    array_of_cells := pbr.ProBufLookupList(pbr.rootcell, [1], 4); // Comes back with a list of all the field 4'a
    try
      // Outer Loop
      i := 0;
      while i < length(array_of_cells) do // Loops over the array_of_cells
      begin
        cp := array_of_cells[i];
        array_of_cells_8 := pbr.ProBufLookupList(cp, [], 8);
        try
          // Inner Loop
          for k := 0 to length(array_of_cells_8) - 1 do // Loop over the second array_of_cells
          begin
            cp8 := array_of_cells_8[k];
            cp2 := pbr.ProBufLookupfield(cp8, [31, 1, 101, 2]);
            if cp2 <> nil then
            begin
              str := pbr.ProBufLookupValue(cp2, [], [11, 11.2, 11.3]);
              if str <> '' then
              begin
                Result := clean_str(str);
                i := length(array_of_cells); // Break the outer loop
                break; // Break the inner loop
              end;
            end;
          end;
          inc(i);
        finally
          array_of_cells_8 := nil;
        end;
      end;
    finally
      array_of_cells := nil;
    end;

  finally
    pbr.free;
  end;
end;

// -----------------------------------------------------------------------------
// Bookmark Artifact
// -----------------------------------------------------------------------------
procedure Bookmark_Artifact_Source(Entry: TEntry; name_str: string; category_str: string; bm_comment_str: string);
var
  bmDeviceEntry: TEntry;
  bmCreate: boolean;
  bmFolder: TEntry;
  device_name_str: string;
begin
  bmCreate := True;
  if assigned(Entry) then
  begin
    bmDeviceEntry := GetDeviceEntry(Entry);
    device_name_str := '';
    if assigned(bmDeviceEntry) then device_name_str := bmDeviceEntry.SaveName + BS;
    bmFolder := FindBookmarkByName('Artifacts' + BS + 'Source Files' + BS {+ device_name_str + BS} + category_str + BS + name_str, bmCreate);
    if assigned(bmFolder) then
    begin
      if not IsItemInBookmark(bmFolder, Entry) then
      begin
        AddItemsToBookmark(bmFolder, DATASTORE_FILESYSTEM, Entry, bm_comment_str);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Bytes To Hex
// -----------------------------------------------------------------------------
function BytesToHex_StartPos(SomeBytes: TBytes; start_pos: integer = 0; end_pos: integer = 2): string;
var
  i: integer;
  aByte: byte;
begin
  Result := '';
  for i := start_pos to length(SomeBytes) - 1 do
  begin
    if i = end_pos then
      break;
    aByte := SomeBytes[i];
    Result := Result + ' ' + IntToHex(aByte, 2);
  end;
end;

// -----------------------------------------------------------------------------
// Bytes To String
// -----------------------------------------------------------------------------
function BytesToStr_StartPos(SomeBytes: TBytes; start_pos: integer = 0): string;
var
  i: integer;
  aByte: byte;
begin
  Result := '';
  for i := start_pos to length(SomeBytes) - 1 do
  begin
    if SomeBytes[i] = $01 then
      break;
    aByte := SomeBytes[i];
    if (aByte < $20) or (aByte in [$7F .. $BF]) then
      aByte := $20;
    Result := Result + string(chr(aByte));
  end;
end;

// -----------------------------------------------------------------------------
// Bytes To Integer
// -----------------------------------------------------------------------------
function BytesToInt(const bytes: array of byte): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 0 to length(bytes) - 1 do
    Result := Result or (bytes[i] shl (8 * i));
end;

// -----------------------------------------------------------------------------
// Column Value By Name As Date Time
// -----------------------------------------------------------------------------
function ColumnValueByNameAsDateTime(Statement: TSQLite3Statement; const colConf: TSQL_Table): TDateTime;
var
  iCol: integer;
begin
  Result := 0;
  iCol := ColumnByName(Statement, copy(colConf.sql_col, 5, Length(colConf.sql_col)));
  if (iCol > -1) then
  begin
    if colConf.read_as = ftLargeInt then
    try
      Result := Int64ToDateTime_ConvertAs(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end
    else if colConf.read_as = ftFloat then
    try
      Result := GHFloatToDateTime(Statement.Columnint64(iCol), colConf.convert_as);
    except
      on e: exception do
      begin
        Progress.Log(e.message);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Create Artifact Description Text
// -----------------------------------------------------------------------------
function Create_Artifact_Description_Text(Item: TSQL_FileSearch): string;
var
  aStringList: TStringList;

  procedure AddIntToSL(const name_str: string; aint: integer);
  begin
    if aint > 0 then
      aStringList.Add(RPad(name_str + ':', 25) + IntToStr(aint));
  end;

  procedure AddStrToSL(const name_str: string;const astr: string);
  begin
    if Trim(astr) <> '' then
      aStringList.Add(RPad(name_str + ':', 25) + astr);
  end;

begin
  Result := '';
  aStringList := TStringList.Create;
  try
    AddStrToSL('Process_ID', Item.fi_Process_ID);
    AddStrToSL('Name_Program', Item.fi_Name_Program);
    AddStrToSL('Name_Program_Type', Item.fi_Name_Program_Type);
    AddStrToSL('Name_OS', Item.fi_Name_OS);
    AddStrToSL('Process_As', Item.fi_Process_As);
    AddIntToSL('Carve_Adjustment', Item.fi_Carve_Adjustment);
    AddStrToSL('Carve_Footer', Item.fi_Carve_Footer);
    AddStrToSL('Carve_Header', Item.fi_Carve_Header);
    AddIntToSL('Icon_Category', Item.fi_Icon_Category);
    AddIntToSL('Icon_OS', Item.fi_Icon_OS);
    AddIntToSL('Icon_Program', Item.fi_Icon_Program);
    AddStrToSL('NodeByName', Item.fi_NodeByName);
    AddStrToSL('Reference_Info', Item.fi_Reference_Info);
    AddStrToSL('Rgx_Itun_Bkup_Dmn', Item.fi_Rgx_Itun_Bkup_Dmn);
    AddStrToSL('Rgx_Itun_Bkup_Nme', Item.fi_Rgx_Itun_Bkup_Nme);
    AddStrToSL('Glob1_Search', Item.fi_Glob1_Search);
    AddStrToSL('Glob2_Search', Item.fi_Glob2_Search);
    AddStrToSL('Glob3_Search', Item.fi_Glob3_Search);
    AddStrToSL('Regex_Search', Item.fi_Regex_Search);
    AddStrToSL('RootNodeName', Item.fi_RootNodeName);
    AddStrToSL('Signature_Parent', Item.fi_Signature_Parent);
    AddStrToSL('Signature_Sub', Item.fi_Signature_Sub);
    AddStrToSL('SQLStatement', Item.fi_SQLStatement);
    AddStrToSL('fi_SQLTables_Required', Item.fi_SQLTables_Required);
    AddStrToSL('SQLPrimary_Tablestr', Item.fi_SQLPrimary_Tablestr);
    Result := aStringList.Text;
  finally
    aStringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// Create Global Search
// -----------------------------------------------------------------------------
procedure Create_Global_Search(anint: integer; Item: TSQL_FileSearch; aStringList: TStringList);
var
  Glob1_Search: string;
  Glob2_Search: string;
  Glob3_Search: string;
begin
  Glob1_Search := Item.fi_Glob1_Search;
  Glob2_Search := Item.fi_Glob2_Search;
  Glob3_Search := Item.fi_Glob3_Search;
  if Trim(Glob1_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob1_Search + ''');'); // noslz
  if Trim(Glob2_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob2_Search + ''');'); // noslz
  if Trim(Glob3_Search) <> '' then aStringList.Add('FindEntries_StringList.Add(''' + Glob3_Search + ''');'); // noslz
  if Trim(Glob1_Search) =  '' then Progress.Log(IntTostr(anint) + ': No Glob Search'); // noslz
end;

// -----------------------------------------------------------------------------
// Determine Then Skip Or Add
// -----------------------------------------------------------------------------
procedure DetermineThenSkipOrAdd(Entry: TEntry; const biTunes_Domain_str: string; const biTunes_Name_str: string);
var
  bm_comment_str: string;
  bmwal_Entry: TEntry;
  DeterminedFileDriverInfo: TFileTypeInformation;
  File_Added_bl: boolean;
  first_Entry: TEntry;
  i: integer;
  Item: TSQL_FileSearch;
  MatchSignature_str: string;
  NowProceed_bl: boolean;
  reason_str: string;
  Reason_StringList: TStringList;
  trunc_EntryName_str: string;

begin
  File_Added_bl := False;
  if Entry.isSystem and ((POS('$I30', Entry.EntryName) > 0) or (POS('$90', Entry.EntryName) > 0)) then
  begin
    NowProceed_bl := False;
    Exit;
  end;

  DeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
  reason_str := '';
  trunc_EntryName_str := copy(Entry.EntryName, 1, 25);
  Reason_StringList := TStringList.Create;
  try
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      NowProceed_bl := False;
      reason_str := '';
      Item := gArr[i];

      // -----------------------------------------------------------------------
      // Special validation (these files are still sent here via the Regex match)
      // -----------------------------------------------------------------------
      if (not NowProceed_bl) then
      begin
        if RegexMatch(Entry.EntryName, gArr[i].fi_Regex_Search, False) and
          (RegexMatch(Entry.EntryName, 'History.mapsdata', False) or
          RegexMatch(Entry.EntryName, '1321e6b74c9dfe411e7e129d6a8ae7cc645af9d0', False) or
          RegexMatch(Entry.EntryName, '^pass\.json', False) or
          RegexMatch(Entry.EntryName, 'flattened-data', False)) then
          NowProceed_bl := True;
      end;

      // Proceed if SubDriver has been identified
      if (not NowProceed_bl) then
      begin
        if Item.fi_Signature_Sub <> '' then
        begin
          if RegexMatch(RemoveSpecialChars(DeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then
          begin
            NowProceed_bl := True;
            reason_str := 'ShortDisplayName = Required SubSig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + Item.fi_Signature_Sub + ')';
            Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(A)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
          end;
        end;
      end;

      // Set the MatchSignature to the parent
      if (not NowProceed_bl) and not(UpperCase(Entry.Extension) = '.JSON') then
      begin
        MatchSignature_str := '';
        if Item.fi_Signature_Sub = '' then
          MatchSignature_str := UpperCase(Item.fi_Signature_Parent);
        // Proceed if SubDriver is blank, but File/Path Name and Parent Signature match
        if ((RegexMatch(Entry.EntryName, Item.fi_Regex_Search, False)) or (RegexMatch(Entry.FullPathName, Item.fi_Regex_Search, False))) and
          ((UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) or (RegexMatch(DeterminedFileDriverInfo.ShortDisplayName, MatchSignature_str, False))) then
        begin
          NowProceed_bl := True;
          reason_str := 'ShortDisplay matches Parent sig:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(B)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      // Proceed if EntryName is unknown, but iTunes Domain, Name and Sig match
      if (not NowProceed_bl) then
      begin
        if RegexMatch(biTunes_Domain_str, Item.fi_Rgx_Itun_Bkup_Dmn, False) and RegexMatch(biTunes_Name_str, Item.fi_Rgx_Itun_Bkup_Nme, False) and (UpperCase(DeterminedFileDriverInfo.ShortDisplayName) = UpperCase(MatchSignature_str)) then
        begin
          NowProceed_bl := True;
          reason_str := 'Proceed on Sig and iTunes Domain\Name:' + SPACE + '(' + DeterminedFileDriverInfo.ShortDisplayName + ' = ' + MatchSignature_str + ')';
          Reason_StringList.Add(format(GFORMAT_STR, ['', 'Added(C)', IntToStr(i), IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]))
        end;
      end;

      if NowProceed_bl then
      begin
        gArr_ValidatedFiles_TList[i].Add(Entry);
        File_Added_bl := True;

        // Bookmark Source Files
        if BL_BOOKMARK_SOURCE then
        begin
          bm_comment_str := Create_Artifact_Description_Text(gArr[i]);
          Bookmark_Artifact_Source(Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
          if assigned(gFileSystemDataStore) then
          begin
            first_Entry := gFileSystemDataStore.First;
            if assigned(first_Entry) then
            begin
              bmwal_Entry := gFileSystemDataStore.FindByPath(first_Entry, Entry.FullPathName + '-wal');
              if bmwal_Entry <> nil then
              begin
                Bookmark_Artifact_Source(bmwal_Entry, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, CATEGORY_NAME, bm_comment_str);
              end;
            end;
          end;
        end;

        // Add Flags
        if BL_USE_FLAGS then Entry.Flags := Entry.Flags + [Flag5]; // Green Flag
      end;
    end;

    Reason_StringList.Add(format(GFORMAT_STR, ['', 'Ignored', '', IntToStr(Entry.ID), DeterminedFileDriverInfo.ShortDisplayName, trunc_EntryName_str, reason_str]));

    if BL_PROCEED_LOGGING then
    begin
      for i := 0 to Reason_StringList.Count - 1 do
        Progress.Log(Reason_StringList[i]);
    end;

  finally
    Reason_StringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// File Sub Signature Match
// -----------------------------------------------------------------------------
function FileSubSignatureMatch(Entry: TEntry): boolean;
var
  i: integer;
  param_num_int: integer;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  Item: TSQL_FileSearch;
begin
  Result := False;
  if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
  begin
    for i := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      Item := gArr[i];
      if Item.fi_Signature_Sub <> '' then
      begin
        aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
        if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
        begin
          if BL_PROCEED_LOGGING then
            Progress.Log(RPad('Proceed' + HYPHEN + 'Identified by SubSig:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
          Result := True;
          break;
        end;
      end;
    end;
  end
  else
  begin
    if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
    begin
      for i := 0 to gParameter_Num_StringList.Count - 1 do
      begin
        if not Progress.isRunning then
          break;
        param_num_int := StrToInt(gParameter_Num_StringList[i]);
        Item := gArr[param_num_int];
        if Item.fi_Signature_Sub <> '' then
        begin
          aDeterminedFileDriverInfo := Entry.DeterminedFileDriverInfo;
          if RegexMatch(RemoveSpecialChars(aDeterminedFileDriverInfo.ShortDisplayName), RemoveSpecialChars(Item.fi_Signature_Sub), False) then // 20-FEB-19 Changed to Regex for multiple sigs
          begin
            if BL_PROCEED_LOGGING then
              Progress.Log(RPad('Proceed' + HYPHEN + 'File Sub-Signature Match:', RPAD_VALUE) + Entry.EntryName + SPACE + 'Bates:' + IntToStr(Entry.ID));
            Result := True;
            break;
          end;
        end;
      end
    end
  end;
end;

// -----------------------------------------------------------------------------
// Get Full Name
// -----------------------------------------------------------------------------
function GetFullName(Item: PSQL_FileSearch): string;
var
  ApplicationName: string;
  TypeName: string;
  OSName: string;
begin
  Result := '';
  ApplicationName := Item.fi_Name_Program;
  TypeName := Item.fi_Name_Program_Type;
  OSName := Item.fi_Name_OS;
  if (ApplicationName <> '') then
  begin
    if (TypeName <> '') then
      Result := format('%0:s %1:s', [ApplicationName, TypeName])
    else
      Result := ApplicationName;
  end
  else
    Result := TypeName;
  if OSName <> '' then
    Result := Result + ' ' + OSName;
end;

// -----------------------------------------------------------------------------
// Length of Array Table
// -----------------------------------------------------------------------------
function LengthArrayTABLE(anArray: TSQL_Table_array): integer;
var
  i: integer;
begin
  Result := 0;
  for i := 1 to 100 do
  begin
    if anArray[i].sql_col = '' then
      break;
    Result := i;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// -----------------------------------------------------------------------------
// Setup Column For Folder
// -----------------------------------------------------------------------------
function SetUpColumnforFolder(aReferenceNumber: integer; anArtifactFolder: TArtifactConnectEntry; out col_DF: TDataStoreFieldArray; ColCount: integer; aItems: TSQL_Table_array): boolean;
var
  col_label: string;
  col_source_created: TDataStoreField;
  col_source_file: TDataStoreField;
  col_source_modified: TDataStoreField;
  col_source_path: TDataStoreField;
  Field: TDataStoreField;
  i: integer;
  Item: TSQL_FileSearch;
  NumberOfColumns: integer;

begin
  Result := True;
  Item := gArr[aReferenceNumber];
  NumberOfColumns := ColCount;
  SetLength(col_DF, ColCount + 1);

  if assigned(anArtifactFolder) then
  begin
    for i := 1 to NumberOfColumns do
    begin
      try
        if not Progress.isRunning then
          Exit;
        Field := gArtifactsDataStore.DataFields.FieldByName(aItems[i].fex_col);
        if assigned(Field) and (Field.FieldType <> aItems[i].col_type) then
        begin
          MessageUser(SCRIPT_NAME + DCR + 'WARNING: New column: ' + DCR + aItems[i].fex_col + DCR + 'already exists as a different type. Creation skipped.');
          Result := False;
        end
        else
        begin
          col_label := '';
          col_DF[i] := gArtifactsDataStore.DataFields.Add(aItems[i].fex_col + col_label, aItems[i].col_type);
          if col_DF[i] = nil then
          begin
            MessageUser(SCRIPT_NAME + DCR + 'Cannot use a fixed field. Please contact support@getdata.com quoting the following error: ' + DCR + SCRIPT_NAME + SPACE + IntToStr(aReferenceNumber) + SPACE + aItems[i].fex_col);
            Result := False;
          end;
        end;
      except
        MessageUser(ATRY_EXCEPT_STR + 'Failed to create column');
      end;
    end;

    // Set the Source Columns --------------------------------------------------
    col_source_file := gArtifactsDataStore.DataFields.GetFieldByName('Source_Name');
    col_source_path := gArtifactsDataStore.DataFields.GetFieldByName('Source_Path');
    col_source_created := gArtifactsDataStore.DataFields.GetFieldByName('Source_Created');
    col_source_modified := gArtifactsDataStore.DataFields.GetFieldByName('Source_Modified');

    // Columns -----------------------------------------------------------------
    if Result then
    begin
      // Enables the change of column headers when switching folders - This is the order of displayed columns
      for i := 1 to NumberOfColumns do
      begin
        if not Progress.isRunning then
          break;
        if aItems[i].Show then
        begin
          // Progress.Log('Add Field Name: ' + col_DF[i].FieldName);
          anArtifactFolder.AddField(col_DF[i]);
        end;
      end;

      if (Item.fi_Process_As = 'POSTPROCESS') then
        anArtifactFolder.AddField(col_source_path)
      else
      begin
        anArtifactFolder.AddField(col_source_file);
        anArtifactFolder.AddField(col_source_path);
        anArtifactFolder.AddField(col_source_created);
        anArtifactFolder.AddField(col_source_modified);
      end;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Name
// -----------------------------------------------------------------------------
function SQLColumnByName(Statement: TSQLite3Statement; const Name: string): integer;
var
  i: integer;
begin
  Result := -1;
  for i := 0 to Statement.ColumnCount do
  begin
    if not Progress.isRunning then
      break;
    if SameText(Statement.ColumnName(i), Name) then
    begin
      Result := i;
      break;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Integer
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsInt(Statement: TSQLite3Statement; const Name: string): integer;
var
  iCol: integer;
begin
  Result := -1;
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnInt(iCol);
end;

// -----------------------------------------------------------------------------
// SQL Column Value - Text
// -----------------------------------------------------------------------------
function SQLColumnValueByNameAsText(Statement: TSQLite3Statement; const Name: string): string;
var
  iCol: integer;
begin
  Result := '';
  iCol := SQLColumnByName(Statement, Name);
  if iCol > -1 then
    Result := Statement.ColumnText(iCol);
end;

// -----------------------------------------------------------------------------
// SQL - Read Artifacts DB
// -----------------------------------------------------------------------------
procedure Read_SQLite_DB();
var
  i: integer;
  sql_db_path_str: string;
  sqlselect: TSQLite3Statement;

begin
  // Locate the SQLite DB
  sql_db_path_str := GetDatabasesDir + 'Artifacts' + BS + ARTIFACTS_DB; // noslz
  if not FileExists(sql_db_path_str) then
  begin
    MessageUser('Locations' + ':' + SPACE + 'Did not locate Artifacts SQLite Database:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
    Exit;
  end;
  Progress.Log(RPad('Found Database:', RPAD_VALUE) + sql_db_path_str);

  // Open the database
  gArtifacts_SQLite := TSQLite3Database.Create;
  try
    gdb_read_bl := False;
    if FileExists(sql_db_path_str) then
    try
      gArtifacts_SQLite.Open(sql_db_path_str);
      gdb_read_bl := True;
      Progress.Log(RPad('Database Read:', RPAD_VALUE) + BoolToStr(gdb_read_bl, True));
    except
      on e: exception do
      begin
        Progress.Log(e.message);
        Exit;
      end;
    end;

    // Get the number of rows in the database as gNumberOfSearchItems
    if gdb_read_bl then
    begin
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT COUNT(*) FROM Artifact_Values');
      try
        gNumberOfSearchItems := 0;
        if sqlselect.Step = SQLITE_ROW then
          gNumberOfSearchItems := sqlselect.ColumnInt(0);
      finally
        sqlselect.free;
      end;
      Progress.Log(RPad('Database Rows:', RPAD_VALUE) + IntToStr(gNumberOfSearchItems));

      if gNumberOfSearchItems = 0 then
      begin
        MessageUser('No records were read from:' + SPACE + sql_db_path_str + '.' + DCR + TSWT);
        Exit;
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      SetLength(gArr, gNumberOfSearchItems);

      // Populate the Array with values from the database
      sqlselect := TSQLite3Statement.Create(gArtifacts_SQLite, 'SELECT * FROM Artifact_Values ORDER BY fi_Name_Program');
      try
        i := 0;
        while (sqlselect.Step = SQLITE_ROW) and (Progress.isRunning) do
        begin
          gArr[i].fi_Carve_Adjustment    := SQLColumnValueByNameAsInt(sqlselect,  'fi_Carve_Adjustment');
          gArr[i].fi_Carve_Footer        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Footer');
          gArr[i].fi_Carve_Header        := SQLColumnValueByNameAsText(sqlselect, 'fi_Carve_Header');
          gArr[i].fi_Glob1_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob1_Search');
          gArr[i].fi_Glob2_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob2_Search');
          gArr[i].fi_Glob3_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Glob3_Search');
          gArr[i].fi_Icon_Category       := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Category');
          gArr[i].fi_Icon_OS             := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_OS');
          gArr[i].fi_Icon_Program        := SQLColumnValueByNameAsInt(sqlselect,  'fi_Icon_Program');
          gArr[i].fi_Name_Program        := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program');
          gArr[i].fi_Name_OS             := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_OS');
          gArr[i].fi_Name_Program_Type   := SQLColumnValueByNameAsText(sqlselect, 'fi_Name_Program_Type');
          gArr[i].fi_NodeByName          := SQLColumnValueByNameAsText(sqlselect, 'fi_NodeByName');
          gArr[i].fi_Process_As          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_As');
          gArr[i].fi_Process_ID          := SQLColumnValueByNameAsText(sqlselect, 'fi_Process_ID');
          gArr[i].fi_Reference_Info      := SQLColumnValueByNameAsText(sqlselect, 'fi_Reference_Info');
          gArr[i].fi_Regex_Search        := SQLColumnValueByNameAsText(sqlselect, 'fi_Regex_Search');
          gArr[i].fi_Rgx_Itun_Bkup_Dmn   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Dmn');
          gArr[i].fi_Rgx_Itun_Bkup_Nme   := SQLColumnValueByNameAsText(sqlselect, 'fi_Rgx_Itun_Bkup_Nme');
          gArr[i].fi_RootNodeName        := SQLColumnValueByNameAsText(sqlselect, 'fi_RootNodeName');
          gArr[i].fi_Signature_Parent    := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Parent');
          gArr[i].fi_Signature_Sub       := SQLColumnValueByNameAsText(sqlselect, 'fi_Signature_Sub');
          gArr[i].fi_SQLPrimary_Tablestr := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLPrimary_Tablestr');
          gArr[i].fi_SQLStatement        := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLStatement');
          gArr[i].fi_SQLTables_Required  := SQLColumnValueByNameAsText(sqlselect, 'fi_SQLTables_Required');
          gArr[i].fi_Test_Data           := SQLColumnValueByNameAsText(sqlselect, 'fi_Test_Data');
          inc(i);
        end;

      finally
        sqlselect.free;
      end;

    end;

  finally
    gArtifacts_SQLite.Close;
    gArtifacts_SQLite.free;
  end;
end;

// -----------------------------------------------------------------------------
// StrippedOfNonAscii
// -----------------------------------------------------------------------------
function StrippedOfNonAscii(const str: string): string;
var
  idx, Count: integer;
begin
  SetLength(Result, Length(str));
  Count := 0;
  for idx := 1 to Length(str) do
  begin
    if ((str[idx] >= #32) and (str[idx] <= #127)) or (str[idx] in [#10, #13]) then
    begin
      Inc(Count);
      Result[Count] := str[idx];
    end;
  end;
  SetLength(Result, Count);
end;

// -----------------------------------------------------------------------------
// Test for Do Process
// -----------------------------------------------------------------------------
function TestForDoProcess(ARefNum: integer): boolean;
begin
  Result := False;

  if gArr[ARefNum].fi_Process_As = 'POSTPROCESS' then
  begin
    Result := True;
    Exit;
  end;

  if (ARefNum <= Length(gArr) - 1) and Progress.isRunning then
  begin
    if (CmdLine.Params.Indexof(IntToStr(ARefNum)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      Progress.Log(RPad('Process List #' + IntToStr(ARefNum) + SPACE + '(' + IntToStr(gArr_ValidatedFiles_TList[ARefNum].Count) + '):', RPAD_VALUE) + gArr[ARefNum].fi_Name_Program + SPACE + gArr[ARefNum].fi_Name_Program_Type + RUNNING);
      if gArr[ArefNum].fi_Process_ID <> '' then  Progress.Log(RPad('Process ID:', RPAD_VALUE) + gArr[ARefNum].fi_Process_ID);
      if gArr[ARefNum].fi_Process_As <> '' then Progress.Log(RPad('fi_Process_As:', RPAD_VALUE) + gArr[ARefNum].fi_Process_As);
      Result := True;
    end;
  {$IF DEFINED (ISFEXGUI)}
  end
  else
  begin
    if not Progress.isRunning then
      Exit;
    Progress.Log('Error: RefNum > ' + IntToStr(Length(gArr) - 1)); // noslz
  {$IFEND}
  end;
end;

// -----------------------------------------------------------------------------
// Total Validated File Count
// -----------------------------------------------------------------------------
function TotalValidatedFileCountInTLists: integer;
var
  i: integer;
begin
  Result := 0;
  for i := 0 to Length(gArr) - 1 do
  begin
    if not Progress.isRunning then
      break;
    Result := Result + gArr_ValidatedFiles_TList[i].Count;
  end;
end;

// =============================================================================
// Do Process
// =============================================================================
procedure DoProcess(anArtifactFolder: TArtifactConnectEntry; ref_num: integer; aItems: TSQL_Table_array);
var
  aArtifactEntry: TEntry;
  ADDList: TList;
  aPropertyTree: TPropertyParent;
  aRootProperty: TPropertyNode;
  carved_str: string;
  CarvedData: TByteInfo;
  CarvedEntry: TEntry;
  col_DF: TDataStoreFieldArray;
  ColCount: integer;
  Display_Name_str: string;
  DNT_sql_col: string;
  end_pos: int64;
  FooterProgress: TPAC;
  g, i, x: integer;
  h_startpos, h_offset, h_count, f_offset, f_count: int64;
  HeaderReader, FooterReader, CarvedEntryReader: TEntryReader;
  HeaderRegex, FooterRegEx: TRegEx;
  Item: PSQL_FileSearch;
  last_scan_dt: TDateTime;
  last_scan_int64: int64;
  last_scan_str: string;
  mac_address_int64: int64;
  mac_address_str: string;
  mydb: TSQLite3Database;
  newEntryReader: TEntryReader;
  newJOURNALReader: TEntryReader;
  newWALReader: TEntryReader;
  Node1: TPropertyNode;
  NodeList1: TObjectList;
  NumberOfNodes: integer;
  Re: TDIPerlRegEx;
  records_read_int: integer;
  SomeBytes: TBytes;
  sql_row_count: integer;
  sqlselect: TSQLite3Statement;
  test_bytes: TBytes;
  TotalFiles: int64;
  variant_Array: array of variant;

  OneByte: byte;
  pos_0a: integer;
  tmp_int64: int64;
  tmp_str: string;
  tmp_dt: TDateTime;

  procedure NullTheArray;
  var
    idx: integer;
  begin
    for idx := 1 to ColCount do
      variant_Array[idx] := null;
  end;

// ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼
  procedure AddToModule;
  var
    NEntry: TArtifactItem;
    IsAllEmpty: boolean;
  begin

    // Do not add when it is a Triage
    if CmdLine.Params.Indexof(TRIAGE) > -1 then
      Exit;

     // Check if all columns are empty
    IsAllEmpty := True;
    for g := 1 to ColCount do
    begin
      if (not VarIsEmpty(variant_Array[g])) and (not VarIsNull(variant_Array[g])) then
      begin
        IsAllEmpty := False;
        break;
      end;
    end;

    // If artifacts are found
    if not IsAllEmpty then
    begin

      // Create the Category Folder to the tree ----------------------------------
      if not assigned(gArtConnect_CatFldr) then
      begin
        gArtConnect_CatFldr := AddArtifactCategory(nil, CATEGORY_NAME, -1, gArr[1].fi_Icon_Category); { Sort index, icon }
        gArtConnect_CatFldr.Status := gArtConnect_CatFldr.Status + [dstUserCreated];
      end;

      // Create artifact sub-folder
      gArtConnect_ProgFldr[ref_num] := AddArtifactConnect(TEntry(gArtConnect_CatFldr),
      Item.fi_Name_Program,
      Item.fi_Name_Program_Type,
      Item.fi_Name_OS,
      Item.fi_Icon_Program,
      Item.fi_Icon_OS);

      // If the artifact sub-folder has been created
      if assigned(gArtConnect_ProgFldr[ref_num]) then
      begin
        // Set the status of the artifacts sub-folder
        gArtConnect_ProgFldr[ref_num].Status := gArtConnect_ProgFldr[ref_num].Status + [dstUserCreated];

        // Setup the columns of the artifact sub-folder
        SetUpColumnforFolder(ref_num, gArtConnect_ProgFldr[ref_num], col_DF, ColCount, aItems);

        // Create the new entry
        NEntry := TArtifactItem.Create;
        NEntry.SourceEntry := aArtifactEntry;
        NEntry.Parent := gArtConnect_ProgFldr[ref_num];
        NEntry.PhysicalSize := 0;
        NEntry.LogicalSize := 0;

        // Populate the columns
        try
          for g := 1 to ColCount do
          begin
            if not Progress.isRunning then
              break;

            if (VarIsNull(variant_Array[g])) or (VarIsEmpty(variant_Array[g])) then
              Continue;

            case col_DF[g].FieldType of
              ftDateTime:
                try
                  col_DF[g].AsDateTime[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftDateTime conversion');
                  end;
                end;

              ftFloat:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsFloat[NEntry] := StrToFloat(variant_Array[g]);
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftFloat conversion');
                    end;
                  end;

              ftInteger:
                try
                  if Trim(variant_Array[g]) = '' then
                    variant_Array[g] := null
                  else
                    col_DF[g].AsInteger[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftInteger conversion');
                  end;
                end;

              ftLargeInt:
                try
                  col_DF[g].AsInt64[NEntry] := variant_Array[g];
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftLargeInt conversion');
                  end;
                end;

              ftString:
                if VarIsStr(variant_Array[g]) and (variant_Array[g] <> '') then
                  try
                    col_DF[g].AsString[NEntry] := variant_Array[g];
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                      Progress.Log(HYPHEN + 'ftString conversion');
                    end;
                  end;

              ftBytes:
                try
                  col_DF[g].AsBytes[NEntry] := variantToArrayBytes(variant_Array[g]);
                except
                  on e: exception do
                  begin
                    Progress.Log(e.message);
                    Progress.Log(HYPHEN + 'ftBytes conversion');
                  end;
                end;

            end;
          end;
        except
          on e: exception do
          begin
            Progress.Log(e.message);
          end;
        end;
        ADDList.Add(NEntry);
      end;
    end;
    NullTheArray;
  end;
// ▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲

  function LatLon_ProtoBuf(bytes: TBytes; type_int: integer) : variant;
  var
    pbr: TGDProtoBufReader;
    lat_str: string;
  begin
    Result := '';
    pbr := TGDProtoBufReader.Create;
    try
      pbr.Loadfrombytes(bytes);
      try
        lat_str := pbr.ProBufLookupValue(pbr.rootcell, [],[type_int]);
        Result := StrToInt(lat_str) / 10000000.0;
      except
      end;
    finally
      pbr.free;
    end
  end;

  function AirTag_LastScan_ProtoBuf(bytes: TBytes) : string;
  var
    pbr: TGDProtoBufReader;
  begin
    Result := '';
    pbr := TGDProtoBufReader.Create;
    try
      pbr.Loadfrombytes(bytes);
      try
        Result := pbr.ProBufLookupValue(pbr.rootcell, [], [1]);
      except
      end;
    finally
      pbr.free;
    end
  end;

  procedure Nodefrog_AirTag_Info(aNode: TPropertyNode);
  const
    ROOTLVL = '\[\d+\]  \{\}';
  var
    i: integer;
    theNodeList: TObjectList;
    parentnode: TPropertyNode;
  begin
    if assigned(aNode) and Progress.isRunning then
    begin
      parentnode := aNode;
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        for i := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          aNode := TPropertyNode(theNodeList.items[i]);
          if assigned(aNode) and assigned(parentnode) then
          begin
            // Must use the parent folder to determine the correct value to add
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'serialNumber') then variant_Array[1] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'identifier') then variant_Array[2] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'owner') then variant_Array[3] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,'partInfo',True) and (aNode.PropName = 'name') then variant_Array[4] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,'productInformation',True) and (aNode.PropName = 'modelName') then variant_Array[5] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,'productInformation',True) and (aNode.PropName = 'manufacturerName') then variant_Array[6] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,'productInformation',True) and (aNode.PropName = 'productIdentifier') then variant_Array[7] := aNode.PropDisplayValue; // noslz
          end;
          Nodefrog_AirTag_Info(aNode);
        end;
      end;
    end;
  end;

  // Do Process - Node Frog - CarPlay ------------------------------------------
  procedure Nodefrog_CarPlay(aNode: TPropertyNode);
  var
    i: integer;
    theNodeList: TObjectList;
    parentnode: TPropertyNode;
  begin
    if assigned(aNode) and Progress.isRunning then
    begin
      parentnode := aNode;
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        for i := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          aNode := TPropertyNode(theNodeList.items[i]);
          if assigned(aNode) and assigned(parentnode) then
          begin
            if (aNode.PropName = 'name') then variant_Array[1] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'bluetoothAddress') then variant_Array[2] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'carplayWiFiUUID') then variant_Array[3] := aNode.PropDisplayValue; // noslz
          end;
          Nodefrog_CarPlay(aNode);
        end;
      end;
    end;
  end;

  procedure Nodefrog_FindMyDevice(aNode: TPropertyNode); // GetData
  const
    ROOTLVL = '\[\d+\]  \{\}';
  var
    i: integer;
    theNodeList: TObjectList;
    parentnode: TPropertyNode;
    timestamp_int64: int64;
  begin
    if assigned(aNode) and Progress.isRunning then
    begin
      parentnode := aNode;
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        for i := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          aNode := TPropertyNode(theNodeList.items[i]);
          if assigned(aNode) and assigned(parentnode) then
          begin
            // Must use the parent folder to determine the correct value to add
            if assigned(parentnode) and RegexMatch(parentnode.PropName, ROOTLVL, True) then variant_Array[1] := Trim(StringReplace(parentnode.PropName,'{}','',[rfReplaceAll])); // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'name') then variant_Array[2] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'deviceDisplayName') then variant_Array[3] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'deviceModel') then variant_Array[4] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'deviceDiscoveryId') then variant_Array[5] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'id') then variant_Array[6] := aNode.PropDisplayValue; // noslz
            if (parentnode.PropName = 'address {}') and (aNode.PropName = 'mapItemFullAddress') then variant_Array[7] := aNode.PropDisplayValue; // noslz
            if (parentnode.PropName = 'address {}') and (aNode.PropName = 'administrativeArea') then variant_Array[8] := aNode.PropDisplayValue; // noslz
            if RegexMatch(parentnode.PropName,'crowdSourcedLocation',True) and (aNode.PropName = 'timeStamp') then  // noslz
            begin
              try
                timestamp_int64 := StrToInt64(aNode.PropDisplayValue);
                variant_Array[9] := Int64ToDateTime_ConvertAs(timestamp_int64, 'UNIX_MS');
              except
                on e: exception do
                begin
                  Progress.Log(e.message);
                  Exit;
                end;
              end;
            end;
            if (parentnode.PropName = 'location {}') and (aNode.PropName = 'latitude') then variant_Array[10] := aNode.PropDisplayValue; // noslz
            if (parentnode.PropName = 'location {}') and (aNode.PropName = 'longitude') then variant_Array[11] := aNode.PropDisplayValue; // noslz
            if (parentnode.PropName = 'crowdSourcedLocation {}') and (aNode.PropName = 'latitude') then variant_Array[12] := aNode.PropDisplayValue; // noslz
            if (parentnode.PropName = 'crowdSourcedLocation {}') and (aNode.PropName = 'longitude') then variant_Array[13] := aNode.PropDisplayValue; // noslz
          end;
          Nodefrog_FindMyDevice(aNode);
        end;
      end;
    end;
  end;

  procedure Nodefrog_FindMyItems(aNode: TPropertyNode);
  const
    ROOTLVL = '\[\d+\]  \{\}';
  var
    i: integer;
    theNodeList: TObjectList;
    parentnode: TPropertyNode;
    timestamp_int64: int64;
  begin
    if assigned(aNode) and Progress.isRunning then
    begin
      parentnode := aNode;
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        for i := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          aNode := TPropertyNode(theNodeList.items[i]);
          if assigned(aNode) and assigned(parentnode) then
          begin
            // Must use the parent folder to determine the correct value to add
            if RegexMatch(parentnode.PropName,'crowdSourcedLocation',True) and (aNode.PropName = 'timeStamp') then  // noslz
            try
              timestamp_int64 := StrToInt64(aNode.PropDisplayValue);
              variant_array[1] := Int64ToDateTime_ConvertAs(timestamp_int64, 'UNIX_MS');
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'serialNumber') then // noslz
            try
              variant_Array[2] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

            if RegexMatch(parentnode.PropName,ROOTLVL,True) and (aNode.PropName = 'owner') then // noslz
            try
              variant_Array[3] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

            if RegexMatch(parentnode.PropName,'partInfo',True) and (aNode.PropName = 'name') then // noslz
            try
              variant_Array[4] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

            if RegexMatch(parentnode.PropName,'productInformation',True) and (aNode.PropName = 'modelName') then // noslz
            try
              variant_Array[5] := aNode.PropDisplayValue;
            except
            end;

            if (parentnode.PropName = 'location {}') and (aNode.PropName = 'latitude') then // noslz
            try
              variant_Array[6] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

            if (parentnode.PropName = 'location {}') and (aNode.PropName = 'longitude') then // noslz
            try
              variant_Array[7] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

//            if (parentnode.PropName = 'address {}') and (aNode.PropName = 'administrativeArea') then // noslz
//            try
//              variant_Array[8] := aNode.PropDisplayValue;
//            except
//              on e: exception do
//              begin
//                Progress.Log(e.message);
//              end;
//            end;

            if (parentnode.PropName = 'address {}') and (aNode.PropName = 'mapItemFullAddress') then // noslz
            try
              variant_Array[8] := aNode.PropDisplayValue;
            except
              on e: exception do
              begin
                Progress.Log(e.message);
              end;
            end;

//            if (parentnode.PropName = 'address {}') and (aNode.PropName = 'subAdministrativeArea') then // noslz
//            try
//              variant_Array[10] := aNode.PropDisplayValue;
//            except
//              on e: exception do
//              begin
//                Progress.Log(e.message);
//              end;
//            end;

          end;
          Nodefrog_FindMyItems(aNode);
        end;
      end;
    end;
  end;
  
  // Do Process - Weather Node Frog --------------------------------------------
  procedure Nodefrog_Weather(aNode: TPropertyNode);
  var
    i: integer;
    theNodeList: TObjectList;
    parentnode: TPropertyNode;
  begin
    if assigned(aNode) and Progress.isRunning then
    begin
      parentnode := aNode;
      theNodeList := aNode.PropChildList;
      if assigned(theNodeList) then
      begin
        for i := 0 to theNodeList.Count - 1 do
        begin
          if not Progress.isRunning then
            break;
          aNode := TPropertyNode(theNodeList.items[i]);
          if assigned(aNode) and assigned(parentnode) then
          begin
            if (aNode.PropName = 'Name') then variant_Array[2] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'Country') then variant_Array[3] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'TimeZone') then variant_Array[4] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'Lat') then variant_Array[5] := aNode.PropDisplayValue; // noslz
            if (aNode.PropName = 'Lon') then variant_Array[6] := aNode.PropDisplayValue; // noslz
          end;
          Nodefrog_Weather(aNode);
        end;
      end;
    end;
  end;  

  function x_ProtoBuf(aPbr: TGDProtoBufReader; aParent: Cellptr; const aStr: string): string;
  var
    Field: Cellptr;
  begin
    Result := '';
    try
      if aStr = 'DNT_LATITUDE' then
      begin
        Field := aPbr.ProBufLookupfield(aParent, [4, 5]);
        if assigned(Field) then
        begin
          Field := aPbr.ProBufLookupfield(aParent, [4, 5, 3]);
          if assigned(Field) then
            Result := Floattostr(extended(aPbr.GetCellInt64(Field)));
        end
        else
        begin
          Field := aPbr.ProBufLookupfield(aParent, [4, 6]);
          if assigned(Field) then
          begin
            Field := aPbr.ProBufLookupfield(aParent, [4, 6, 1, 3]);
            if assigned(Field) then
              Result := Floattostr(extended(aPbr.GetCellInt64(Field)));
          end;
        end;
      end
      else if aStr = 'DNT_LONGITUDE' then
      begin
        Field := aPbr.ProBufLookupfield(aParent, [4, 5]);
        if assigned(Field) then
        begin
          Field := aPbr.ProBufLookupfield(aParent, [4, 5, 4]);
          if assigned(Field) then
            Result := Floattostr(extended(aPbr.GetCellInt64(Field)));
        end
        else
        begin
          Field := aPbr.ProBufLookupfield(aParent, [4, 6]);
          if assigned(Field) then
          begin
            Field := aPbr.ProBufLookupfield(aParent, [4, 6, 1, 2]);
            if assigned(Field) then
              Result := Floattostr(extended(aPbr.GetCellInt64(Field)));
          end;
        end;
      end
        else if astr = 'DNT_PLACE'     then Result := aPbr.ProBufLookupValue(aParent, [4],[1])
        else if astr = 'DNT_TIMESTAMP' then Result := aPbr.ProBufLookupValue(aParent, [],[2])
        else if astr = 'DNT_URL'       then Result := aPbr.ProBufLookupValue(aParent, [4],[13]);
    except
      on e: exception do
        Progress.Log(e.message);
    end;
  end;

  procedure MapTilesSQLBlob(records_read_int: integer); // https://www.digitalforensics.io/ios-forensics-vmp4-file-format/
  const
    BL_LOG = False;
  var
    amem_strm1: TMemoryStream;
    decompstrm: TZDecompressionStream;
    msg_str: string;
    tempmem: TMemoryStream;
    test_bytes: TBytes;	
    vpm4_compression_flag_bl: boolean;
    vpm4_s2_address_str: string;
    vpm4_s2_fldtyp_int: word;
    vpm4_s2_length: integer;
    vpm4_s2_offset: integer;

  begin
    try
      if BL_LOG then Progress.Log('Record#:' + SPACE + IntToStr(records_read_int));

      // Attempt to retrieve blob data
      try
        test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
      except
        on e: Exception do
        begin
          if BL_LOG then
            Progress.Log(e.Message);
          Exit;
        end;
      end;

      // Validate data length
      if Length(test_bytes) < 20 then
      begin
        if BL_LOG then
          Progress.Log('MapTilesSQLBlob: Insufficient data length.'); // nols
        Exit;
      end;

      if Length(test_bytes) >= 20 then
      begin
        // Turn test_bytes into a memory stream
        if BL_LOG then Progress.Log(RPad('Length test_bytes:', RPAD_VALUE) + inttostr(Length(test_bytes)));

        tempmem := TMemoryStream.Create;
        try
          // Write test_bytes into a memory stream
          tempmem.Write(@test_bytes[0], Length(test_bytes));
          tempmem.Position := 0;

          // Read section 2 and determine the field type
          vpm4_s2_fldtyp_int := PWord(@test_bytes[18])^;

          if BL_LOG then Progress.Log(RPad('Field Type Int Value:', RPAD_VALUE) + IntToStr(vpm4_s2_fldtyp_int));

          if (vpm4_s2_fldtyp_int = 10) or (vpm4_s2_fldtyp_int = 11) then
          begin
            if BL_LOG then Progress.Log(RPad('VMP4 Header:', RPAD_VALUE) + Trim(BytesToHex_StartPos(test_bytes, 0, 20)));

            // Retrieve the field 2 starting offset
            vpm4_s2_offset := PInteger(@test_bytes[20])^;
            if BL_LOG then Progress.Log(RPad(HYPHEN + 'S2 Off:', RPAD_VALUE) + IntToStr(vpm4_s2_offset));

            // Determine compression flag
            vpm4_compression_flag_bl := Boolean(test_bytes[vpm4_s2_offset]);
            if BL_LOG then Progress.Log(RPad(HYPHEN + 'S2 Compression Flag:', RPAD_VALUE) + BoolToStr(vpm4_compression_flag_bl, True));

            // Get the length of field 2
            vpm4_s2_length := PInteger(@test_bytes[24])^;
            if BL_LOG then Progress.Log(RPad(HYPHEN + 'S2 Length:', RPAD_VALUE) + IntToStr(vpm4_s2_length));

            // Handle non-compressed data
            if not vpm4_compression_flag_bl then
            begin
              vpm4_s2_address_str := Trim(BytesToStr_StartPos(test_bytes, vpm4_s2_offset));
              if BL_LOG then Progress.Log(RPad(HYPHEN + 'S2 Str:', RPAD_VALUE) + Trim(vpm4_s2_address_str));
              variant_array(g) := vpm4_s2_address_str;
            end
            else
            // Handle compressed data
            begin
              tempmem.Position := vpm4_s2_offset + 5;
              decompstrm := TZDecompressionStream.Create(tempmem);
              try
                amem_strm1 := TMemoryStream.Create;
                try
                  if BL_LOG then Progress.Log(RPad(HYPHEN + 'S3 Size of compressed:', RPAD_VALUE) + IntToStr(decompstrm.Size - vpm4_s2_offset));

                  amem_strm1.CopyFrom(decompstrm, decompstrm.Size);
                  if BL_LOG then Progress.Log(RPad(HYPHEN + 'S3 Size of uncompressed:', RPAD_VALUE) + IntToStr(amem_strm1.Size));

                  amem_strm1.Position := 0;
                  SetLength(SomeBytes, amem_strm1.Size);
                  amem_strm1.Read(SomeBytes[0], amem_strm1.Size);

                  if BL_LOG then Progress.Log(RPad('Output (BytesToStr):', RPAD_VALUE) + Trim(BytesToStr_StartPos(SomeBytes, 0)));
                  if BL_LOG then Progress.Log(RPad('Output (UTF8 BytesToStr):', RPAD_VALUE) + UTF8BytesToString(SomeBytes, 0, Length(SomeBytes)));

                  msg_str := UTF8BytesToString(SomeBytes, 0, Length(SomeBytes));
                  variant_array[g] := msg_str;
                finally
                  amem_strm1.Free;
                end;
              finally
                decompstrm.Free;
              end;
            end;
          end;

        finally
          tempmem.Free;
        end;

      end;
    except
      on e: Exception do
      begin
        if BL_LOG then
          Progress.Log(e.Message);
      end;
    end;
  end;

  // Do Process - Test SQL -----------------------------------------------------
  function Test_SQL_Tables(refnum: integer): boolean;
  var
    LineList: TStringList;
    idx: integer;
    sql_tbl_count: integer;
    sqlselect_row: TSQLite3Statement;
    s: integer;
    temp_sql_str: string;

  begin
    Result := True;

    // Table count check to eliminate corrupt SQL files with no tables
    sql_tbl_count := 0;
    try
      sqlselect := TSQLite3Statement.Create(mydb, 'SELECT name FROM sqlite_master WHERE type=''table''');
      try
        while sqlselect.Step = SQLITE_ROW do
        begin
          if not Progress.isRunning then
            break;
          sql_tbl_count := sql_tbl_count + 1;
        end;
      finally
        FreeAndNil(sqlselect);
      end;
    except
      Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
      Result := False;
      Exit;
    end;

    if sql_tbl_count = 0 then
    begin
      Result := False;
      Exit;
    end;

    // Table count check to eliminate SQL files without required tables
    if (sql_tbl_count > 0) and (gArr[refnum].fi_SQLTables_Required <> '') then
    begin
      LineList := TStringList.Create;
      LineList.Delimiter := ','; // Default, but ";" is used with some locales
      LineList.StrictDelimiter := True; // Required: strings are separated *only* by Delimiter
      LineList.Sorted := True;
      LineList.Duplicates := dupIgnore;
      try
        LineList.DelimitedText := gArr[refnum].fi_SQLTables_Required;
        if LineList.Count > 0 then
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[0]) + '''))';
          for idx := 0 to LineList.Count - 1 do
            temp_sql_str := temp_sql_str + ' or (upper(name) = upper(''' + Trim(LineList[idx]) + '''))';

          temp_sql_str := temp_sql_str + ')';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) and (sqlselect_row.ColumnInt(0) = LineList.Count) then
                sql_tbl_count := 1
              else
              begin
                Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + aArtifactEntry.EntryName, RPAD_VALUE) + 'Ignored (Found ' + IntToStr(sqlselect_row.ColumnInt(0)) + ' of ' + IntToStr(LineList.Count) + ' required tables).');
                Result := False;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            MessageUser(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
            Result := False;
          end;
        end;

        // Log the missing required tables
        for s := 0 to LineList.Count - 1 do
        begin
          temp_sql_str := 'select count(*) from sqlite_master where type = ''table'' and ((upper(name) = upper(''' + Trim(LineList[s]) + ''')))';
          try
            sqlselect_row := TSQLite3Statement.Create(mydb, temp_sql_str);
            try
              if (sqlselect_row.Step = SQLITE_ROW) then
              begin
                if sqlselect_row.ColumnInt(0) = 0 then
                begin
                  Progress.Log(RPad('', RPAD_VALUE) + HYPHEN + 'Missing table:' + SPACE + Trim(LineList[s]));
                end;
              end;
            finally
              sqlselect_row.free;
            end;
          except
            Progress.Log(ATRY_EXCEPT_STR + 'An exception has occurred (usually means that the SQLite file is corrupt).');
          end;
        end;

      finally
        LineList.free;
      end;
    end;

    // Row Count the matched table
    if sql_tbl_count > 0 then
    begin
      sql_row_count := 0;
      sqlselect_row := TSQLite3Statement.Create(mydb, 'SELECT COUNT(*) FROM ' + gArr[refnum].fi_SQLPrimary_Tablestr);
      try
        while sqlselect_row.Step = SQLITE_ROW do
        begin
          sql_row_count := sqlselect_row.ColumnInt(0);
          break;
        end;
      finally
        sqlselect_row.free;
      end;
    end;
  end;

  // DoProcess - Tile X, Y, to Lat, Lon ----------------------------------------
  function TileXYToLatLong(x, Y: integer; Zoom: integer; out Latitude, Longitude: Double): boolean;
  var
    n: Double;
  begin
    try
      n := Power(2.0, Zoom);
      Longitude := x / n * 360.0 - 180.0;
      Latitude := ArcTan(Sinh(Pi * (1 - 2 * Y / n))) * (180.0 / Pi);
      Result := True;
    except
      on e: exception do
      begin
        Result := False;
      end;
    end;
  end;

  // DoProcess - Python - Execute To Result StringList -------------------------
  function Python_Execute_To_StringList(anEntry: TEntry; python_name_str: string; Python_Result_StringList: TStringList) : boolean;
  var
    dt_str: string;
    Error_str: string;
    ExitCode_crd: cardinal;
    ExportEntryReader: TEntryReader;
    ExportFile: TFileStream;
    FileSavename_str: string;
    Output_str: string;
    python_code_pth: string;
    select_str: string;
    sqlselect: TSQLite3Statement;
    theCommand_str: string;
    tmp_name_str: string;
    tmp_StringList: TStringList;

  begin
    Result := False;
    Progress.DisplayMessageNow := 'Opening SQLite database' + RUNNING;
    if FileExists(gPythonDB_pth) then
    begin
      Progress.Log(RPad('Found Python DB:', RPAD_VALUE) + gPythonDB_pth);
      gPython_SQLite := TSQLite3Database.Create;
      if assigned(gPython_SQLite) then
      try
        tmp_StringList := TStringList.Create;
        gPython_SQLite.Open(gPythonDB_pth);
        try
          select_str := 'SELECT * FROM Python_Scripts'; // noslz
          try
            sqlselect := TSQLite3Statement.Create(gPython_SQLite, select_str);
            try
              while sqlselect.Step = SQLITE_ROW do
              begin
                if not Progress.isRunning then break;
                tmp_name_str := SQLColumnValueByNameAsText(sqlselect, 'Python_Name'); // noslz
                if tmp_name_str = python_name_str then
                begin
                  tmp_StringList.Text := SQLColumnValueByNameAsText(sqlselect, 'Python_Code'); // noslz
                  if ForceDirectories(GetExportedDir + 'Temp\Python') then // noslz
                  try
                    python_code_pth := GetExportedDir + 'Temp\Python' + BS + tmp_name_str + '.py';
                    try
                      tmp_StringList.SaveToFile(python_code_pth); // noslz
                    except
                      on e: exception do
                      begin
                        Progress.Log(e.message);
                      end;
                    end;
                    if FileExists(python_code_pth) then
                    begin
                      Progress.Log(RPad('Wrote temp py file:', RPAD_VALUE) + python_code_pth);
                      Result := True;
                    end;
                  except
                    on e: exception do
                    begin
                      Progress.Log(e.message);
                    end;
                  end;
                end;
              end;
            finally
              sqlselect.free;
            end;
          except
            on e: exception do
              Progress.Log(e.message);
          end;
        finally
          gPython_SQLite.Close;
        end;

        // If the code was saved, now save the target file to dis
        if FileExists(python_code_pth) then
        begin
          ExportEntryReader := TEntryReader.Create;
          try
            if ExportEntryReader.OpenData(anEntry) then
            begin
              try
                FileSavename_str := ExtractFilePath(python_code_pth) + anEntry.EntryName + '_BATES#_' + IntToStr(anEntry.ID);
                ExportFile := TFileStream.Create(FileSavename_str, fmOpenWrite or fmCreate);
                try
                  ExportFile.CopyFrom(ExportEntryReader, anEntry.LogicalSize);
                  Progress.Log(RPad('Export to RamDisk as:', RPAD_VALUE) + FileSavename_str);
                finally
                  ExportFile.free;
                  ExportFile := nil;
                end;
              except
                Progress.Log(ATRY_EXCEPT_STR + 'Error in Exporting file: ' + anEntry.EntryName);
              end;
            end;
          finally
            ExportEntryReader.free;
          end;
        end;

        if FileExists(python_code_pth) and FileExists(FileSavename_str) then
        begin
          theCommand_str := 'cmd.exe /c ""' +
          gportablepython_path_str + BS + 'python.exe' + '" "' +
          python_code_pth + '" "' +
          FileSavename_str + '"';
          CaptureConsoleOutput('', theCommand_str, Output_str, Error_str, ExitCode_crd, 0, 50000);
          Progress.Log(RPad('CMD String:', RPAD_VALUE) + theCommand_str);
          Progress.Log(RPad('Exit Code:', RPAD_VALUE) + IntToStr(ExitCode_crd));
          if Trim(Error_str) <> '' then Progress.Log(RPad('Error:', RPAD_VALUE) + Error_str);

          if Trim(Output_str) <> '' then
            Python_Result_StringList.Text := Output_str;

        end;
      finally
        FreeAndNil(gPython_SQLite);
        tmp_StringList.free;
      end;
    end;
  end;

  // DoProcess - Unix Micro Seconds to Date Time -------------------------------
  function UnixMicrosecondsToDateTime(const Microseconds: int64): TDateTime;
  const
    UnixStartDate: TDateTime = 25569; // December 30, 1899 is day 0, January 1, 1970 is 25569
    MicrosecondsPerDay: int64 = 24 * 60 * 60 * 1000000;
  begin
    Result := Microseconds / MicrosecondsPerDay + UnixStartDate;
  end;



// Start of Do Process =========================================================
var
  col_str: string;
  h: integer;
  index: integer;
  lat_dbl: double;
  lat_str: string;
  lon_dbl: double;
  lon_str: string;
  lyft_timestamp_int64: int64;
  lyft_tmp_str: string;
  Node_Weather_Last_Updated: TPropertyNode;
  pb_map_item_storage_str: string;
  pb_route_request_storage_str: string;
  pbr: TGDProtoBufReader;
  pbrField: CellPtr;
  pbrFields : TArrayCells;
  process_proceed_bl: boolean;
  Python_Result_StringList: TStringList;
  SearchData: TSearchResult;
  success_bl: boolean;
  temp_flt: Double;
  temp_process_counter: integer;
  tempstr: string;
  xcrd_str: string;
  ycrd_str: string;
  zlvl_str: string;
  zmapitemsstorage_bytes: TBytes;
  zroutereqeststorage_bytes: TBytes;

begin
  if gArtifactsDataStore = nil then
    Exit;

  Item := @gArr[ref_num];
  temp_process_counter := 0;
  ColCount := LengthArrayTABLE(aItems);
  if (gArr_ValidatedFiles_TList[ref_num].Count > 0) then
  begin
    begin
      process_proceed_bl := True;
      if process_proceed_bl then
      begin
        SetLength(variant_Array, ColCount + 1);
        ADDList := TList.Create;
        newEntryReader := TEntryReader.Create;
        try
          Progress.Max := gArr_ValidatedFiles_TList[ref_num].Count;
          Progress.DisplayMessageNow := 'Process' + SPACE + PROGRAM_NAME + ' - ' + Item.fi_Name_OS + RUNNING;
          Progress.CurrentPosition := 1;

          // Regex Setup -------------------------------------------------------
          CarvedEntryReader := TEntryReader.Create;
          FooterProgress := TPAC.Create;
          FooterProgress.Start;
          FooterReader := TEntryReader.Create;
          FooterRegEx := TRegEx.Create;
          FooterRegEx.CaseSensitive := True;
          FooterRegEx.Progress := FooterProgress;
          FooterRegEx.SearchTerm := Item.fi_Carve_Footer;
          HeaderReader := TEntryReader.Create;
          HeaderRegex := TRegEx.Create;
          HeaderRegex.CaseSensitive := True;
          HeaderRegex.Progress := Progress;
          HeaderRegex.SearchTerm := Item.fi_Carve_Header;

          if HeaderRegex.LastError <> 0 then
          begin
            Progress.Log('HeaderRegex Error: ' + IntToStr(HeaderRegex.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          if FooterRegEx.LastError <> 0 then
          begin
            Progress.Log(RPad('!!!!!!! FOOTER REGEX ERROR !!!!!!!:', RPAD_VALUE) + IntToStr(FooterRegEx.LastError));
            aArtifactEntry := nil;
            Exit;
          end;

          TotalFiles := gArr_ValidatedFiles_TList[ref_num].Count;
          // Loop Validated Files ----------------------------------------------
          for i := 0 to gArr_ValidatedFiles_TList[ref_num].Count - 1 do { addList is freed a the end of this loop }
          begin
            if not Progress.isRunning then
              break;

            Progress.IncCurrentprogress;
            temp_process_counter := temp_process_counter + 1;
            Display_Name_str := GetFullName(Item);
            Progress.DisplayMessageNow := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + RUNNING;
            aArtifactEntry := TEntry(gArr_ValidatedFiles_TList[ref_num].items[i]);

            if assigned(aArtifactEntry) and newEntryReader.OpenData(aArtifactEntry) and (newEntryReader.Size > 0) and Progress.isRunning then
            begin
              if BL_USE_FLAGS then aArtifactEntry.Flags := aArtifactEntry.Flags + [Flag8]; // Gray Flag = Process Routine

              // ===============================================================
              // Custom Processing
              // ===============================================================
              if gArr[Ref_Num].fi_Process_ID = 'LOC_AND_GOOGLEMAPS_SEARCHES' then
              begin
                if newEntryReader.OpenData(aArtifactEntry) and (aArtifactEntry.LogicalSize > 0) then
                begin
                  begin
                    newEntryReader.Position := 0;
                    pos_0a := 0;
                    for x := 0 to aArtifactEntry.LogicalSize - 1 do
                    begin
                      if not Progress.isRunning then
                        break;
                      newEntryReader.Read(OneByte, 1);
                      if OneByte = $0A then
                      begin
                        pos_0a := x;
                        Break;
                      end;
                    end;

                    if pos_0a > 0 then
                    begin
                      newEntryReader.Position := pos_0a;
                      setlength(SomeBytes, aArtifactEntry.LogicalSize - pos_0a);
                      newEntryReader.Read(SomeBytes[0], aArtifactEntry.LogicalSize - pos_0a);
                      pbr := TGDProtoBufReader.Create;
                      try
                        pbr.Loadfrombytes(SomeBytes);
                        pbrFields := pbr.ProBufLookupList(pbr.rootcell, [], 1);
                        if length(pbrFields) > 0 then
                        begin
                          Progress.Log(' number pbr records ' + IntToStr(high(pbrFields)));
                          for pbrField in pbrFields do
                          begin
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := aItems[g].sql_col;
                              tmp_str := '';
                              tmp_int64 := 0;
                              tmp_dt := null;

                              if DNT_sql_col = 'DNT_TIMESTAMP' then
                              begin
                                tmp_str := x_ProtoBuf(pbr, pbrField, 'DNT_TIMESTAMP');
                                if RegexMatch(tmp_str, '\d{16}', False) then
                                  try
                                    tmp_int64 := StrToInt64(tmp_str);
                                    tmp_dt := UnixMicrosecondsToDateTime(tmp_int64);
                                    variant_Array[g] := tmp_dt;
                                  except
                                  end;
                              end;

                              if DNT_sql_col = 'DNT_PLACE' then
                                variant_Array[g] := x_ProtoBuf(pbr, pbrField, 'DNT_PLACE');

                              if DNT_sql_col = 'DNT_LATITUDE' then
                                try
                                  variant_Array[g] := x_ProtoBuf(pbr, pbrField, 'DNT_LATITUDE');
                                  // Progress.Log('DNT_LATITUDE: ' + tmp_str);
                                except
                                end;

                              if DNT_sql_col = 'DNT_LONGITUDE' then
                                try
                                  variant_Array[g] := x_ProtoBuf(pbr, pbrField, 'DNT_LONGITUDE');
                                  // Progress.Log('DNT_LONGITUDE: ' + tmp_str);
                                except
                                end;

                              if DNT_sql_col = 'DNT_URL' then
                              begin
                                tmp_str := x_ProtoBuf(pbr, pbrField, 'DNT_URL');
                                if (tmp_str <> '') and (tmp_str <> '0') then
                                  variant_Array[g] := tmp_str;
                              end;

                            end;
                            AddToModule;
                          end; // for pbrFields
                        end;
                      finally
                        pbr.free;
                      end;
                    end;
                  end;
                end;
              end;

              if gArr[Ref_Num].fi_Process_ID = 'LOC_AND_AIRTAG_PERSONAL_SAFETY_LAST_SCAN' then
              begin
                if newEntryReader.OpenData(aArtifactEntry) and (aArtifactEntry.LogicalSize > 0) then
                begin
                  begin
                    for g := 1 to ColCount do
                    begin
                      DNT_sql_col := aItems[g].sql_col;
                      delete(DNT_sql_col, 1, 4);
                      if DNT_sql_col = 'LAST_SCAN_ANDROID' then
                      begin
                        // Progress.Log(newEntryReader.AsHexString(7));
                        newEntryReader.Position := 0;
                        setlength(SomeBytes, 7);
                        newEntryReader.Read(SomeBytes[0], 7);
                        last_scan_str := AirTag_LastScan_ProtoBuf(SomeBytes);
                        if RegexMatch(last_scan_str, '\d{13}', False) then
                        begin
                          try
                            last_scan_int64 := StrToInt64(last_scan_str);
                            last_scan_dt := Int64ToDateTime_ConvertAs(last_scan_int64, 'UNIX_MS');
                            variant_Array[g] := last_scan_dt;
                          except
                          end;
                        end;
                      end;
                      AddToModule;
                    end;
                  end;
                end;
              end;

              // ===============================================================
              // PROCESS AS: Nodes - .PropName, .PropDisplayValue, .PropValue, .PropDataType
              // ===============================================================
              if ((UpperCase(Item.fi_Process_As) = PROCESS_AS_PLIST) or (UpperCase(Item.fi_Process_As) = PROCESS_AS_XML)) and
              ((UpperCase(Item.fi_Signature_Parent) = 'PLIST (BINARY)') or
              (UpperCase(Item.fi_Signature_Parent) = 'JSON') or
              (UpperCase(Item.fi_Signature_Parent) = 'XML')) then
              begin
                aPropertyTree := nil;
                try
                  if ProcessMetadataProperties(aArtifactEntry, aPropertyTree, newEntryReader) and assigned(aPropertyTree) then
                  begin
                    // Set Root Property (Level 0)
                    aRootProperty := aPropertyTree;
                    // Progress.Log(format('%-21s %-26s %-10s %-10s %-20s ',['Number of root child nodes:',IntToStr(aRootProperty.PropChildList.Count),'Bates:',IntToStr(aArtifactEntry.ID),aArtifactEntry.EntryName]));
                    if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                    begin
                      if not Progress.isRunning then
                        break;

                      // Locate NodeByName
                      if CompareText(aRootProperty.PropName, Item.fi_NodeByName) = 0 then
                        Node1 := aRootProperty
                      else
                        Node1 := aRootProperty.GetFirstNodeByName(Item.fi_NodeByName);

                      if assigned(Node1) then
                      begin

                        // Count the number of nodes
                        NodeList1 := Node1.PropChildList;
                        if assigned(NodeList1) then
                          NumberOfNodes := NodeList1.Count
                        else
                          NumberOfNodes := 0;
                        Progress.Log(RPad('Number of child nodes:', RPAD_VALUE) + IntToStr(NumberOfNodes));

                        // AirTags Info IOS;
                        if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_AIRTAG_INFO') and (UpperCase(gArr[Ref_Num].fi_Name_OS) = UpperCase(IOS)) then // noslz
                        begin
                          for i := 0 to NodeList1.Count - 1 do
                          begin
                            Node1 := TPropertyNode(NodeList1.items[i]);
                            if assigned(Node1) then
                            begin
                              Nodefrog_AirTag_Info(Node1);
                              AddToModule;
                            end;
                          end;
                        end;

                        // Find My Items IOS;
                        if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_CARPLAY') and (UpperCase(gArr[Ref_Num].fi_Name_OS) = UpperCase(IOS)) then // noslz
                        begin
                          for i := 0 to NodeList1.Count - 1 do
                          begin
                            Node1 := TPropertyNode(NodeList1.items[i]);
                            if assigned(Node1) then
                            begin
                              Nodefrog_Carplay(Node1);
                              AddToModule;
                            end;
                          end;
                        end;

                        // Find My Device IOS;
                        if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_FINDMYDEVICE') and (UpperCase(gArr[Ref_Num].fi_Name_OS) = UpperCase(IOS)) then // noslz
                        begin
                          for i := 0 to NodeList1.Count - 1 do
                          begin
                            Node1 := TPropertyNode(NodeList1.items[i]);
                            if assigned(Node1) then
                            begin
                              Nodefrog_FindMyDevice(Node1);
                              AddToModule;
                            end;
                          end;
                        end;

                        // Find My Items IOS;
                        if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_FINDMYITEMS') and (UpperCase(gArr[Ref_Num].fi_Name_OS) = UpperCase(IOS)) then // noslz
                        begin
                          for i := 0 to NodeList1.Count - 1 do
                          begin
                            Node1 := TPropertyNode(NodeList1.items[i]);
                            if assigned(Node1) then
                            begin
                              Nodefrog_FindMyItems(Node1);
                              AddToModule;
                            end;
                          end;
                        end;

                        // Weather IOS;
                        if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_WEATHER') and (UpperCase(gArr[Ref_Num].fi_Name_OS) = UpperCase(IOS)) then // noslz
                        begin
                          Node1 := TPropertyNode(aRootProperty.GetFirstNodeByName(Item.fi_NodeByName));
                          if assigned(Node1) then
                          begin
                            for i := 0 to NodeList1.Count - 1 do
                            begin
                              Node_Weather_Last_Updated := TPropertyNode(aRootProperty.GetFirstNodeByName('LastUpdated')); // noslz
                              if assigned(Node_Weather_Last_Updated) then
                              try
                                variant_Array[1] := VarToDateTime(Node_Weather_Last_Updated.PropDisplayValue); // noslz
                              except
                              end;

                              Node1 := TPropertyNode(NodeList1.items[i]);
                              if assigned(Node1) then
                              begin
                                Nodefrog_Weather(Node1);
                                AddToModule;
                              end;
                            end;
                          end;
                        end;

                      end;

                    end
                    else
                      Progress.Log((format('%-39s %-10s %-10s', ['Could not open data:', '-', 'Bates: ' + IntToStr(aArtifactEntry.ID) + ' ' + aArtifactEntry.EntryName])));
                  end;
                finally
                  if assigned(aPropertyTree) then
                    FreeAndNil(aPropertyTree);
                end;
              end;

              // ===============================================================
              // PROCESS AS: - Python
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_PYTHON) then
              begin
                Python_Result_StringList := TStringList.Create;
                try
                  if gArr[Ref_Num].fi_Process_ID = 'LOC_IOS_LYFT_USER' then
                  begin
                    Python_Execute_To_StringList(aArtifactEntry, 'Artifacts_Lyft', Python_Result_StringList);
                    if Trim(Python_Result_StringList.Text) <> '' then
                    begin
                      for g := 1 to ColCount do
                      begin
                        DNT_sql_col := aItems[g].sql_col;
                        delete(DNT_sql_col, 1, 4);
                        for x := 0 to Python_Result_StringList.Count - 1 do
                        begin
                          if RegexMatch(Python_Result_StringList[x],DNT_sql_col,False) then
                          begin
                            lyft_tmp_str := Trim(StringReplace(Python_Result_StringList[x], DNT_sql_col + ':', '', [rfReplaceAll]));
                            if DNT_sql_col = 'EMAILVERIFIEDDATE' then
                            try
                              lyft_timestamp_int64 := StrToInt64(Trim(StringReplace(Python_Result_StringList[x], DNT_sql_col + ':', '', [rfReplaceAll])));
                              variant_Array[g] := Int64ToDateTime_ConvertAs(lyft_timestamp_int64, 'UNIX_MS');
                            except
                            end
                            else
                              variant_Array[g] := lyft_tmp_str;
                            Progress.Log(DNT_sql_col + '<<>>' + variant_Array[g]);
                            Break;
                          end;
                        end;
                      end;
                    end;
                  end;
                finally
                  Python_Result_StringList.free;
                end;
                AddToModule;
              end;

              // ===============================================================
              // PROCESS AS: REGEX CARVE
              // ===============================================================
              if (UpperCase(Item.fi_Process_As) = PROCESS_AS_CARVE) then
              begin
                if HeaderReader.OpenData(aArtifactEntry) and FooterReader.OpenData(aArtifactEntry) then
                  try
                    Progress.Initialize(aArtifactEntry.PhysicalSize, 'Searching ' + IntToStr(i + 1) + ' of ' + IntToStr(TotalFiles) + ' files: ' + aArtifactEntry.EntryName);
                    h_startpos := 0;
                    HeaderReader.Position := 0;
                    HeaderRegex.Stream := HeaderReader;
                    FooterRegEx.Stream := FooterReader;
                    // Find the first match, h_offset returned is relative to start pos
                    HeaderRegex.Find(h_offset, h_count);
                    while h_offset <> -1 do // h_offset returned as -1 means no hit
                    begin
                      // Now look for a footer
                      FooterRegEx.Stream.Position := h_startpos + h_offset + Item.fi_Carve_Adjustment;
                      // Limit looking for the footer to our max size
                      end_pos := h_startpos + h_offset + MAX_CARVE_SIZE;
                      if end_pos >= HeaderRegex.Stream.Size then
                        end_pos := HeaderRegex.Stream.Size - 1; // Don't go past the end!
                      FooterRegEx.Stream.Size := end_pos;
                      FooterRegEx.Find(f_offset, f_count);
                      // Found a footer and the size was at least our minimum
                      if (f_offset <> -1) and (f_offset + f_count >= MIN_CARVE_SIZE) then
                      begin
                        // Footer found - Create an Entry for the data found
                        CarvedEntry := TEntry.Create;
                        // ByteInfo is data described as a list of byte runs, usually just one run
                        CarvedData := TByteInfo.Create;
                        // Point to the data of the file
                        CarvedData.ParentInfo := aArtifactEntry.DataInfo;
                        // Adds the block of data
                        CarvedData.RunLstAddPair(h_offset + h_startpos, f_offset + f_count);
                        CarvedEntry.DataInfo := CarvedData;
                        CarvedEntry.LogicalSize := CarvedData.Datasize;
                        CarvedEntry.PhysicalSize := CarvedData.Datasize;
                        begin
                          if CarvedEntryReader.OpenData(CarvedEntry) then
                          begin
                            CarvedEntryReader.Position := 0;
                            carved_str := CarvedEntryReader.AsPrintableChar(CarvedEntryReader.Size);
                            for g := 1 to ColCount do
                              variant_Array[g] := null; // Null the array
                            // Setup the PerRegex Searches
                            Re := TDIPerlRegEx.Create(nil);
                            Re.CompileOptions := [coCaseLess];
                            Re.SetSubjectStr(carved_str);

                            // Process Google Maps iOS =========================
                            if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_GOOGLEMAPS_CARVE') then
                            begin
                              Re.MatchPattern := '(?<=https:\/\/maps.google.com\/maps\/dir\/\/)(.*?)(?=\/data)'; // noslz
                              if Re.Match(0) >= 0 then
                              begin
                                variant_Array[1] := Re.matchedstr;
                                variant_Array[2] := h_startpos + h_offset;
                              end;
                              AddToModule;
                            end;

                            // Process Google Maps Tiles Carve iOS =============
                            if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_GOOGLEMAPS_TILES_CARVE') then
                            begin
                              zlvl_str := Trim(PerlMatch('(?<=!1i)(\d+)(?=!2i)', carved_str));
                              xcrd_str := Trim(PerlMatch('(?<=!2i)(\d+)(?=!3i)', carved_str));
                              ycrd_str := Trim(PerlMatch('(?<=!3i)(\d+)(?=!4i)', carved_str));
                              if (Trim(zlvl_str) <> '') and (Trim(xcrd_str) <> '') and (Trim(ycrd_str) <> '') then
                              begin
                                success_bl := TileXYToLatLong(StrToInt(xcrd_str), StrToInt(ycrd_str), StrToInt(zlvl_str), lat_dbl, lon_dbl);
                                if success_bl then
                                begin
                                  lat_str := FloatToStr(lat_dbl);
                                  lon_str := FloatToStr(lon_dbl);
                                end
                                else
                                  Progress.Log('Failed to calculate latitude and longitude: Google Maps Tiles Carve iOS'); // noslz
                              end;
                              for g := 1 to ColCount do
                              begin
                                DNT_sql_col := aItems[g].sql_col;
                                if (DNT_sql_col = 'DNT_ZOOM') then variant_Array[g] := zlvl_str;
                                if (DNT_sql_col = 'DNT_XCOORDINATE') then variant_Array[g] := xcrd_str;
                                if (DNT_sql_col = 'DNT_YCOORDINATE') then variant_Array[g] := ycrd_str;
                                if (DNT_sql_col = 'DNT_LATITUDE') then variant_Array[g] := lat_str;
                                if (DNT_sql_col = 'DNT_LONGITUDE') then variant_Array[g] := lon_str;
                              end;
                              AddToModule;
                            end;
                            if assigned(Re) then
                              FreeAndNil(Re);
                            AddToModule;
                          end;
                        end;
                      end;
                      HeaderRegex.FindNext(h_offset, h_count); // Find each subsequent header match
                    end;
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error processing ' + aArtifactEntry.EntryName);
                  end;
              end;

              // ===============================================================
              // PROCESS AS: SQL
              // ===============================================================
              if RegexMatch(Item.fi_Signature_Parent, RS_SIG_SQLITE, False) then
              begin
                newWALReader := GetWALReader(gFileSystemDataStore, aArtifactEntry);
                newJOURNALReader := GetJOURNALReader(gFileSystemDataStore, aArtifactEntry);
                try
                  mydb := TSQLite3Database.Create;
                  try
                    mydb.OpenStream(newEntryReader, newJOURNALReader, newWALReader);

                    if Test_SQL_Tables(ref_num) then
                    begin
                      records_read_int := 0;

                      // ***************************************************************************************************************************
                      sqlselect := TSQLite3Statement.Create(mydb, (Item.fi_SQLStatement));
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      //Progress.Log(Item.fi_SQLStatement);
                      //Progress.Log(StringOfChar('~', CHAR_LENGTH));
                      // ***************************************************************************************************************************

                      while sqlselect.Step = SQLITE_ROW do
                      begin
                        if not Progress.isRunning then
                          break;
                        records_read_int := records_read_int + 1;

                        // Progress for large files
                        if sql_row_count > 15000 then
                        begin
                          Progress.DisplayMessages := 'Processing' + SPACE + Display_Name_str + ' (' + IntToStr(temp_process_counter) + ' of ' + IntToStr(gArr_ValidatedFiles_TList[ref_num].Count) + ')' + SPACE +
                            IntToStr(records_read_int) + '/' + IntToStr(sql_row_count) + RUNNING;
                        end;

                        // Read the values from the SQL tables -----------------
                        for g := 1 to ColCount do
                        begin
                          if not Progress.isRunning then
                            break;
                          DNT_sql_col := copy(aItems[g].sql_col, 5, Length(aItems[g].sql_col));

                          // Skip
                          if (UpperCase(Item.fi_Process_ID) = 'LOC_IOS_APPLE_MAPS_TILES') and
                           (DNT_sql_col = 'DATA') and
                           (aItems[g].convert_as = 'BlobBytes') then // noslz
                          begin
                            // PALOFF - Skips the SQL process and uses special processing below
                          end

                          // Skip
                          else if (UpperCase(Item.fi_Process_ID) = 'LOC_IOS_APPLE_MAPS_SEARCHES') and
                          ((DNT_sql_col = 'ZROUTEREQUESTSTORAGE') or (DNT_sql_col = 'ZLATITUDE') or (DNT_sql_col = 'ZLONGITUDE')) then
                          begin
                            // PALOFF - Skips the SQL process and uses special processing below
                          end

                          else if aItems[g].read_as = ftString then
                            variant_Array[g] := ColumnValueByNameAsText(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftinteger) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftinteger) and (aItems[g].col_type = ftString) then
                            variant_Array[g] := ColumnValueByNameAsInt(sqlselect, DNT_sql_col)

                          else if (aItems[g].read_as = ftLargeInt) and (aItems[g].col_type = ftLargeInt) then
                            variant_Array[g] := ColumnValueByNameAsint64(sqlselect, DNT_sql_col)

                          else if (aItems[g].col_type = ftDateTime) then
                          begin
                            if (ColumnValueByNameAsint64(sqlselect, aItems[g].field_name) > 0) then
                            begin
                              try
                                if aItems[g].read_as = ftFloat then
                                begin
                                  temp_flt := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                                  variant_Array[g] := temp_flt;
                                  variant_Array[g] := GHFloatToDateTime(variant_Array[g], aItems[g].convert_as);
                                end
                                else
                                begin
                                  variant_Array[g] := ColumnValueByNameAsint64(sqlselect, aItems[g].field_name);
                                end;
                              except
                                on e: exception do
                                  Progress.Log(e.message);
                              end;
                            end;
                          end;

                          // Add the table name and row location
                          if DNT_sql_col = 'SQLLOCATION' then
                            variant_Array[g] := 'Table: ' + lowercase(Item.fi_SQLPrimary_Tablestr) + ' (row ' + format('%.*d', [4, records_read_int]) + ')'; // noslz

                          // SPECIAL CONVERSIONS FOLLOW ========================

                          // Airtag PersonalSaftey Android - Blob Bytes ======== https://www.digitalforensics.io/ios-forensics-vmp4-file-format/
                          if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_AND_AIRTAG_PERSONAL_SAFETY') and (DNT_sql_col = 'LOCATIONSCAN') then // noslz
                          begin
                            test_bytes := ColumnValueByNameAsBlobBytes(sqlselect, DNT_sql_col);
                            if length(test_bytes) = 28 then
                            begin
                              try
                                if (aItems[g].convert_as = 'DNT_LAT') then variant_array(g) := FloatToStr(LatLon_ProtoBuf(test_bytes,4)); // Lat protobuf
                                if (aItems[g].convert_as = 'DNT_LON') then variant_array(g) := FloatToStr(LatLon_ProtoBuf(test_bytes,5)); // Lon protobuf
                              except
                              end;
                            end;
                          end;

                          // Life360
                          if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_AND_LIFE360_LOCATIONS') then // and (DNT_sql_col = 'DATA') then // noslz
                          begin
                            tempstr := ColumnValueByNameAsText(sqlselect, 'DATA');
                            for g := 1 to ColCount do
                            begin
                              DNT_sql_col := aItems[g].sql_col;
                              if DNT_sql_col = 'DNT_LIFE360ID' then variant_array(g) := StringReplace(trim(PerlMatch('(?<="id":")(.*)(?=")', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_LONGITUDE' then variant_array(g) := StringReplace(trim(PerlMatch('(?<="longitude":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_LATITUDE'  then variant_array(g) := StringReplace(trim(PerlMatch('(?<="latitude":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_ALTITUDE'  then variant_array(g) := StringReplace(trim(PerlMatch('(?<="altitude":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_SPEED'     then variant_array(g) := StringReplace(trim(PerlMatch('(?<="speed":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_COURSE'    then variant_array(g) := StringReplace(trim(PerlMatch('(?<="course":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_BEARING'   then variant_array(g) := StringReplace(trim(PerlMatch('(?<="bearing":)(.*)(?=,)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_BSSID'     then variant_array(g) := StringReplace(trim(PerlMatch('(?<="BSSID":")(.*)(?=",)', tempstr)),'"','',[rfReplaceAll]);
                              if DNT_sql_col = 'DNT_SSID'      then variant_array(g) := StringReplace(trim(PerlMatch('(?<="SSID":")(.*)(?=",)', tempstr)),'"','',[rfReplaceAll]);
                            end;
                          end;

                          // Apple Maps - Searches
                          if UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_APPLE_MAPS_SEARCHES' then // noslz
                          begin
                            pb_route_request_storage_str := '';
                            if (DNT_sql_col = 'ZROUTEREQUESTSTORAGE') then // noslz
                            try
                              zroutereqeststorage_bytes := ColumnValueByNameAsBlobBytes(sqlselect, 'ZROUTEREQUESTSTORAGE'); // noslz
                              SearchData := Apple_Maps_ProtoBuf_Search_Data(zroutereqeststorage_bytes, 'ZROUTEREQUESTSTORAGE', 0); // noslz
                              if (SearchData.Address <> '') and (SearchData.Latitude <> 0) and (SearchData.Longitude <> 0) then
                              begin
                                for h := 1 to ColCount do
                                begin
                                  col_str := aItems[h].sql_col;
                                  if not Progress.isRunning then break;
                                  if col_str = 'DNT_ZROUTEREQUESTSTORAGE' then variant_array(h) := SearchData.Address; // noslz
                                  if col_str = 'DNT_ZLATITUDE' then variant_array(h) := FloatToStr(SearchData.Latitude); // noslz
                                  if col_str = 'DNT_ZLONGITUDE' then variant_array(h) := FloatToStr(SearchData.Longitude); // noslz
                                end;
                              end;
                            except
                              on e: exception do
                              begin
                                Progress.Log(e.message);
                              end;
                            end;
                          end;

                          // Apple Maps - Tiles - Blob Bytes ----------------------------- https://www.digitalforensics.io/ios-forensics-vmp4-file-format/
                          if (UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_APPLE_MAPS_TILES') then
                          begin
                            MapTilesSQLBlob(records_read_int);
                          end;

                          // Apple Maps - Trips
                          if UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_APPLE_MAPS_TRIPS' then
                          begin
                            pb_map_item_storage_str := '';
                            if (DNT_sql_col = 'ZMAPITEMSTORAGE') then
                            try
                              zmapitemsstorage_bytes := ColumnValueByNameAsBlobBytes(sqlselect, 'ZMAPITEMSTORAGE');
                              pb_map_item_storage_str := Apple_Maps_ProtoBuf_Trip_Data(zmapitemsstorage_bytes, 'ZMAPITEMSTORAGE');
                              Progress.Log(IntToStr(records_read_int) + ' >>> ' + pb_map_item_storage_str);
                            except
                            end;
                          end;

                          // Wifi Locations ------------------------------------
                          if UpperCase(gArr[Ref_Num].fi_Process_ID) = 'LOC_IOS_WIFI_LOCATIONS' then // noslz
                          begin
                            if (DNT_sql_col = 'MAC') then // noslz
                            begin
                              mac_address_str := Trim(variant_Array[g]);
                              if length(mac_address_str) >= 12 then
                                try
                                  mac_address_int64 := StrToInt64(mac_address_str);
                                  variant_Array[g] := IntToHex(mac_address_int64, 12);
                                except
                                  Progress.Log(ATRY_EXCEPT_STR + 'creating MAC address.');
                                end;
                            end;
                          end;
                        end;
                        AddToModule;
                      end;
                      if assigned(sqlselect) then
                        sqlselect.free;
                      if BL_PROCEED_LOGGING then
                        Progress.Log(RPad(HYPHEN + 'Bates:' + SPACE + IntToStr(aArtifactEntry.ID) + HYPHEN + copy(aArtifactEntry.EntryName, 1, 40), RPAD_VALUE) + format('%-5s %-5s', [IntToStr(records_read_int), '(SQL)']));
                    end;
                  finally
                    FreeAndNil(mydb);
                    if assigned(newWALReader) then
                      FreeAndNil(newWALReader);
                    if assigned(newJOURNALReader) then
                      FreeAndNil(newJOURNALReader);
                  end;
                except
                  on e: exception do
                  begin
                    Progress.Log(ATRY_EXCEPT_STR + RPad('WARNING:', RPAD_VALUE) + 'ERROR DOING SQL QUERY!');
                    Progress.Log(e.message);
                  end;
                end;
              end;

            end;
            Progress.IncCurrentprogress;
          end;

          // Add to the gArtifactsDataStore
          if assigned(ADDList) and Progress.isRunning then
          begin
            if (CmdLine.Params.Indexof(TRIAGE) = -1) then
              gArtifactsDataStore.Add(ADDList);
            Progress.Log(RPad('Total Artifacts:', RPAD_VALUE) + IntToStr(ADDList.Count));

            // Export L01 files where artifacts are found (controlled by Artifact_Utils.pas)
            if (gArr_ValidatedFiles_TList[ref_num].Count > 0) and (ADDList.Count > 0) then
              ExportToL01(gArr_ValidatedFiles_TList[ref_num], Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type);

            ADDList.Clear;
          end;

          Progress.Log(StringOfChar('-', CHAR_LENGTH));
        finally
          records_read_int := 0;
          FreeAndNil(CarvedEntryReader);
          FreeAndNil(FooterProgress);
          FreeAndNil(FooterReader);
          FreeAndNil(FooterRegEx);
          FreeAndNil(HeaderReader);
          FreeAndNil(HeaderRegex);
          FreeAndNil(ADDList);
          FreeAndNil(newEntryReader);
        end;
      end;

    end;

  end
  else
  begin
    Progress.Log(RPad('', RPAD_VALUE) + 'No files to process.');
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
  end;

end;

// ==========================================================================================================================================================
// Start of Script
// ==========================================================================================================================================================
const
  CREATE_GLOB_SEARCH_BL = False;
  SEARCHING_FOR_ARTIFACT_FILES_STR = 'Searching for Artifact files';

var
  AboutToProcess_StringList: TStringList;
  aDeterminedFileDriverInfo: TFileTypeInformation;
  FolderEntry: TEntry;
  AllFoundList_count: integer;
  AllFoundListUnique: TUniqueListOfEntries;
  DeleteFolder_display_str: string;
  DeleteFolder_TList: TList;
  Enum: TEntryEnumerator;
  ExistingFolders_TList: TList;
  FieldItunesDomain: TDataStoreField;
  FieldItunesName: TDataStoreField;
  FindEntries_StringList: TStringList;
  FoundList, AllFoundList: TList;
  GlobSearch_StringList: TStringList;
  i: integer;
  Item: TSQL_FileSearch;
  iTunes_Domain_str: string;
  iTunes_Name_str: string;
  Process_ID_str: string;
  ResultInt: integer;

{$IF DEFINED (ISFEXGUI)}
  Display_StringList: TStringList;
  MyScriptForm: TScriptForm;
{$IFEND}

procedure LogExistingFolders(aExistingFolders_TList: TList; aDeleteFolder_TList: TList); // Compare About to Process Folders with Existing Artifact Module Folders
const
  LOG_BL = False;
var
  aFolderEntry: TEntry;
  idx, t: integer;
begin
  if LOG_BL then Progress.Log('Existing Folders:');
  for idx := 0 to aExistingFolders_TList.Count - 1 do
  begin
    if not Progress.isRunning then break;
    aFolderEntry := (TEntry(aExistingFolders_TList[idx]));
    if LOG_BL then
      Progress.Log(HYPHEN + aFolderEntry.FullPathName);
    for t := 0 to AboutToProcess_StringList.Count - 1 do
    begin
      if aFolderEntry.FullPathName = AboutToProcess_StringList[t] then
        aDeleteFolder_TList.Add(aFolderEntry);
    end;
  end;
  if LOG_BL then
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

// Start of script -------------------------------------------------------------
var
  anEntry: TEntry;
  anint: integer;
  base_path_str: string;
  param_str: string;
  progress_program_str: string;
  ref_num: integer;
  regex_search_str: string;
  n, r, s: integer;
  temp_int: integer;
  test_param_int: integer;
  temp_process_counter: integer;

begin
  base_path_str := ExtractFileDir(Excludetrailingpathdelimiter(GetStartupDir));
  gportablepython_path_str := base_path_str + BS + 'PortablePython'; // noslz
  if not FileExists(gportablepython_path_str) then
    Progress.Log(RPad('Did not locate:', RPAD_VALUE) + gportablepython_path_str);

  gPythonDB_pth := GetDatabasesDir + 'Python.db'; // noslz
  if not FileExists(gPythonDB_pth) then
    Progress.Log(RPad('Did not locate:', RPAD_VALUE) + gPythonDB_pth);

  Read_SQLite_DB;

  SetLength(gArr_ValidatedFiles_TList, gNumberOfSearchItems);
  SetLength(gArtConnect_ProgFldr, gNumberOfSearchItems);

  Progress.Log(SCRIPT_NAME + ' started' + RUNNING);
  if (CmdLine.Params.Indexof(TRIAGE) > -1) then
    Progress.DisplayTitle := 'Triage' + HYPHEN + 'File System' + HYPHEN + CATEGORY_NAME
  else
    Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME;
  Progress.LogType := ltVerbose; // ltOff, ltVerbose, ltDebug, ltTechnical

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := ('Processing complete.');
    Exit;
  end;

  // Parameters Received -------------------------------------------------------
  param_str := '';
  test_param_int := 0;
  gParameter_Num_StringList := TStringList.Create;
  try
//    if CmdLine.ParamCount = 0 then
//    begin
//      MessageUser('No processing parameters received.');
//      Exit;
//    end
//    else
    begin
      Progress.Log(RPad('Parameters Received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
      for n := 0 to CmdLine.ParamCount - 1 do
      begin
        if not Progress.isRunning then
          break;

        param_str := param_str + '"' + CmdLine.Params[n] + '"' + ' ';
        // Validate Parameters
        if (RegexMatch(CmdLine.Params[n], '\d{1,2}$', False)) then
        begin
          try
            test_param_int := StrToInt(CmdLine.Params[n]);
            if (test_param_int <= -1) or (test_param_int > Length(gArr) - 1) then
            begin
              MessageUser('Invalid parameter received: ' + (CmdLine.Params[n]) + CR + ('Maximum is: ' + IntToStr(Length(gArr) - 1) + DCR + SCRIPT_NAME + ' will terminate.'));
              Exit;
            end;
            gParameter_Num_StringList.Add(CmdLine.Params[n]);
            Item := gArr[test_param_int];
            Progress.Log(RPad(HYPHEN + 'param_str ' + IntToStr(n) + ' = ' + CmdLine.Params[n], RPAD_VALUE) + format('%-10s %-10s %-25s %-12s', ['Ref#: ' + CmdLine.Params[n], CATEGORY_NAME,
              Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type, Item.fi_Name_OS]));
          except
            begin
              Progress.Log(ATRY_EXCEPT_STR + 'Error validating parameter. ' + SCRIPT_NAME + ' will terminate.');
              Exit;
            end;
          end;
        end;
      end;
      Trim(param_str);
    end;
    Progress.Log(RPad('param_str:', RPAD_VALUE) + param_str); // noslz
    Progress.Log(StringOfChar('-', CHAR_LENGTH));

    // Progress Bar Text
    progress_program_str := '';
    if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
    begin
      progress_program_str := PROGRAM_NAME;
    end;

    // Create GlobSearch -------------------------------------------------------
    if CREATE_GLOB_SEARCH_BL then
    begin
      GlobSearch_StringList := TStringList.Create;
      GlobSearch_StringList.Sorted := True;
      GlobSearch_StringList.Duplicates := dupIgnore;
      try
        for n := 0 to Length(gArr) - 1 do
          Create_Global_Search(n, gArr[n], GlobSearch_StringList);
        Progress.Log(GlobSearch_StringList.Text);
      finally
        GlobSearch_StringList.free;
      end;
      Exit;
    end;

    for n := 0 to Length(gArr) - 1 do
    begin
      if not Progress.isRunning then
        break;
      if (CmdLine.Params.Indexof(IntToStr(n)) > -1) then
      begin
        Item := gArr[n];
        progress_program_str := 'Ref#:' + SPACE + param_str + SPACE + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type;
        break;
      end;
    end;

    if progress_program_str = '' then
      progress_program_str := 'Other';

    gArtifactsDataStore := GetDataStore(DATASTORE_ARTIFACTS);
    if not assigned(gArtifactsDataStore) then
    begin
      Progress.Log(DATASTORE_ARTIFACTS + ' module not located.' + DCR + TSWT);
      Exit;
    end;

    try
      gFileSystemDataStore := GetDataStore(DATASTORE_FILESYSTEM);
      if not assigned(gFileSystemDataStore) then
      begin
        Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + DCR + TSWT);
        Exit;
      end;

      try
        FieldItunesDomain := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_DOMAIN);
        FieldItunesName := gFileSystemDataStore.DataFields.FieldByName(FBN_ITUNES_BACKUP_NAME);

        // Create TLists For Valid Files
        for n := 0 to Length(gArr) - 1 do
        begin
          if not Progress.isRunning then
            break;
          gArr_ValidatedFiles_TList[n] := TList.Create;
        end;

        try
          AboutToProcess_StringList := TStringList.Create;
          try
            if (CmdLine.Params.Indexof('MASTER') = -1) then
            begin
              if (CmdLine.Params.Indexof('NOSHOW') = -1) then
              begin
                // PROCESSALL - Create the AboutToProcess_StringList
                if (CmdLine.Params.Indexof(PROCESSALL) > -1) or (CmdLine.ParamCount = 0) then
                begin
                  for n := 0 to Length(gArr) - 1 do
                  begin
                    if not Progress.isRunning then break;
                    Item := gArr[n];
                    AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[n]));
                    Progress.Log(format('%-4s %-59s %-30s', ['#' + IntToStr(n), Item.fi_Process_ID, Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type]));
                  end;
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end
                else

                // PROCESS INDIVIDUAL - Create the AboutToProcess_StringList
                begin
                  if assigned(gParameter_Num_StringList) and (gParameter_Num_StringList.Count > 0) then
                  begin
                    for n := 0 to gParameter_Num_StringList.Count - 1 do
                    begin
                      if not Progress.isRunning then break;
                      temp_int := StrToInt(gParameter_Num_StringList[n]);
                      Item := gArr[temp_int];
                      AboutToProcess_StringList.Add(CATEGORY_NAME + BS + GetFullName(@gArr[temp_int]));
                    end;
                  end;
                end;

                if CmdLine.ParamCount = 0 then
                begin
                  Progress.Log('No parameters received. Use a number or "PROCESSALL".');
                  Exit;
                end;

                // Show the form
                ResultInt := 1; // Continue AboutToProcess

{$IF DEFINED (ISFEXGUI)}
                Display_StringList := TStringList.Create;
                Display_StringList.Sorted := True;
                Display_StringList.Duplicates := dupIgnore;
                try
                  for i := 0 to AboutToProcess_StringList.Count - 1 do
                  begin
                    Display_StringList.Add(ExtractFileName(AboutToProcess_StringList[i]));
                  end;

                  if assigned(AboutToProcess_StringList) and (AboutToProcess_StringList.Count > 30) then
                  begin
                    MyScriptForm := TScriptForm.Create;
                    try
                      MyScriptForm.SetCaption(SCRIPT_DESCRIPTION);
                      MyScriptForm.SetText(Display_StringList.Text);
                      ResultInt := idCancel;
                      if MyScriptForm.ShowModal then
                        ResultInt := idOk
                    finally
                      FreeAndNil(MyScriptForm);
                    end;
                  end
                  else
                  begin
                    ResultInt := MessageBox('Extract Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME, (MB_OKCANCEL or MB_ICONQUESTION or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                  end;
                finally
                  Display_StringList.free;
                end;
{$IFEND}
              end;

              if ResultInt > 1 then
              begin
                Progress.Log(CANCELED_BY_USER);
                Progress.DisplayMessageNow := CANCELED_BY_USER;
                Exit;
              end;

            end;

            // Deal with Existing Artifact Folders
            ExistingFolders_TList := TList.Create;
            DeleteFolder_TList := TList.Create;
            try
              if gArtifactsDataStore.Count > 1 then
              begin
                anEntry := gArtifactsDataStore.First;
                while assigned(anEntry) and Progress.isRunning do
                begin
                  if anEntry.isDirectory then
                    ExistingFolders_TList.Add(anEntry);
                  anEntry := gArtifactsDataStore.Next;
                end;
                gArtifactsDataStore.Close;
              end;

              LogExistingFolders(ExistingFolders_TList, DeleteFolder_TList);

              // Create the delete folder TList and display string
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                Progress.Log('Replacing folders:');
                for s := 0 to DeleteFolder_TList.Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  FolderEntry := TEntry(DeleteFolder_TList[s]);
                  DeleteFolder_display_str := DeleteFolder_display_str + #13#10 + FolderEntry.FullPathName;
                  Progress.Log(HYPHEN + FolderEntry.FullPathName);
                end;
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
              end;

              // Message Box - When artifact folder is already present ---------------------
{$IF DEFINED (ISFEXGUI)}
              if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
              begin
                if (CmdLine.Params.Indexof('MASTER') = -1) and (CmdLine.Params.Indexof('NOSHOW') = -1) then
                begin

                  if (CmdLine.Params.Indexof('NOSHOW') = -1) then
                  begin
                    if assigned(DeleteFolder_TList) and (DeleteFolder_TList.Count > 0) then
                    begin
                      ResultInt := MessageBox('Artifacts have already been processed:' + #13#10 + DeleteFolder_display_str + DCR + 'Replace the existing Artifacts?', 'Extract Artifacts' + HYPHEN + PROGRAM_NAME,
                        (MB_OKCANCEL or MB_ICONWARNING or MB_DEFBUTTON2 or MB_SETFOREGROUND or MB_TOPMOST));
                    end;
                  end
                  else
                    ResultInt := idOk;
                  case ResultInt of
                    idOk:
                      begin
                        try
                          gArtifactsDataStore.Remove(DeleteFolder_TList);
                        except
                          MessageBox('ERROR: There was an error deleting existing artifacts.' + #13#10 + 'Save then reopen your case.', SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
                        end;
                      end;
                    idCancel:
                      begin
                        Progress.Log(CANCELED_BY_USER);
                        Progress.DisplayMessageNow := CANCELED_BY_USER;
                        Exit;
                      end;
                  end;
                end;
              end;
{$IFEND}

            finally
              ExistingFolders_TList.free;
              DeleteFolder_TList.free;
            end;
          finally
            AboutToProcess_StringList.free;
          end;

          // Find iTunes Backups and run Signature Analysis
          if (CmdLine.Params.Indexof('NOITUNES') = -1) then
            Itunes_Backup_Signature_Analysis;

          // Create the RegEx Search String
          regex_search_str := '';
          begin
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if (CmdLine.Params.Indexof(IntToStr(n)) > -1) or (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                if Item.fi_Regex_Search <> ''      then regex_search_str := regex_search_str + '|' + Item.fi_Regex_Search;
                if Item.fi_Rgx_Itun_Bkup_Dmn <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Dmn;
                if Item.fi_Rgx_Itun_Bkup_Nme <> '' then regex_search_str := regex_search_str + '|' + Item.fi_Rgx_Itun_Bkup_Nme;
              end;
            end;
          end;
          if (regex_search_str <> '') and (regex_search_str[1] = '|') then
            Delete(regex_search_str, 1, 1);

          AllFoundList := TList.Create;
          try
            AllFoundListUnique := TUniqueListOfEntries.Create;
            FoundList := TList.Create;
            FindEntries_StringList := TStringList.Create;
            FindEntries_StringList.Sorted := True;
            FindEntries_StringList.Duplicates := dupIgnore;
            try
              if (CmdLine.Params.Indexof(PROCESSALL) > -1) then
              begin
                Progress.Log('Find files by path (PROCESSALL)' + RUNNING);
                for i := low(gArr) to high(gArr) do
                begin
                  if (gArr[i].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob1_Search);
                  if (gArr[i].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob2_Search);
                  if (gArr[i].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[i].fi_Glob3_Search);
                end;
              end
              else
              begin
                Progress.Log('Find files by path (Individual)' + RUNNING);
                for i := 0 to gParameter_Num_StringList.Count - 1 do
                begin
                  anint := StrToInt(gParameter_Num_StringList[i]);
                  if (gArr[anint].fi_Glob1_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob1_Search);
                  if (gArr[anint].fi_Glob2_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob2_Search);
                  if (gArr[anint].fi_Glob3_Search <> '') then FindEntries_StringList.Add(gArr[anint].fi_Glob3_Search);
                end;
              end;

              // Find the files by path and add to AllFoundListUnique
              Progress.Initialize(FindEntries_StringList.Count, STR_FILES_BY_PATH + RUNNING);
              gtick_foundlist_i64 := GetTickCount;
              for i := 0 to FindEntries_StringList.Count - 1 do
              begin
                if not Progress.isRunning then
                  break;
                try
                  Find_Entries_By_Path(gFileSystemDataStore, FindEntries_StringList[i], FoundList, AllFoundListUnique);
                except
                  Progress.Log(RPad(ATRY_EXCEPT_STR, RPAD_VALUE) + 'Find_Entries_By_Path');
                end;
                Progress.IncCurrentprogress;
              end;

              Progress.Log(RPad(STR_FILES_BY_PATH + SPACE + '(Unique)' + COLON, RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Add > 40 char files with no extension to Unique List (possible iTunes backup files)
              Add40CharFiles(AllFoundListUnique);
              Progress.Log(StringOfChar('-', CHAR_LENGTH));

              // Move the AllFoundListUnique list into a TList
              if assigned(AllFoundListUnique) and (AllFoundListUnique.Count > 0) then
              begin
                Enum := AllFoundListUnique.GetEnumerator;
                while Enum.MoveNext do
                begin
                  anEntry := Enum.Current;
                  AllFoundList.Add(anEntry);
                end;
              end;

              // Now work with the TList from now on
              if assigned(AllFoundList) and (AllFoundList.Count > 0) then
              begin
                Progress.Log(RPad('Unique Files by path:', RPAD_VALUE) + IntToStr(AllFoundListUnique.Count));
                Progress.Log(StringOfChar('-', CHAR_LENGTH));

                // Add flags
                if BL_USE_FLAGS then
                begin
                  Progress.Log('Adding flags' + RUNNING);
                  for i := 0 to AllFoundList.Count - 1 do
                  begin
                    if not Progress.isRunning then
                      break;
                    anEntry := TEntry(AllFoundList[i]);
                    anEntry.Flags := anEntry.Flags + [Flag7];
                  end;
                  Progress.Log('Finished adding flags' + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Determine signature
                if assigned(AllFoundList) and (AllFoundList.Count > 0) then
                  SignatureAnalysis(AllFoundList);

                gtick_foundlist_str := (RPad(STR_FILES_BY_PATH + COLON, RPAD_VALUE) + CalcTimeTaken(gtick_foundlist_i64));
              end;

            finally
              FoundList.free;
              AllFoundListUnique.free;
              FindEntries_StringList.free;
            end;

            if assigned(AllFoundList) and (AllFoundList.Count > 0) then
            begin
              Progress.Log(SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              Progress.Log(format(GFORMAT_STR, ['', 'Action', 'Ref#', 'Bates', 'Signature', 'Filename (trunc)', 'Reason'])); // noslz
              Progress.Log(format(GFORMAT_STR, ['', '------', '----', '-----', '---------', '----------------', '------'])); // noslz
              if not BL_PROCEED_LOGGING then
                Progress.Log('BL_PROCEED_LOGGING = False'); // noslz

              AllFoundList_count := AllFoundList.Count;
              Progress.Initialize(AllFoundList_count, SEARCHING_FOR_ARTIFACT_FILES_STR + ':' + SPACE + progress_program_str + RUNNING);
              for i := 0 to AllFoundList.Count - 1 do
              begin
                Progress.DisplayMessages := SEARCHING_FOR_ARTIFACT_FILES_STR + SPACE + '(' + IntToStr(i) + ' of ' + IntToStr(AllFoundList_count) + ')' + RUNNING;
                if not Progress.isRunning then
                  break;
                anEntry := TEntry(AllFoundList[i]);

                if (i mod 10000 = 0) and (i > 0) then
                begin
                  Progress.Log('Processing: ' + IntToStr(i) + ' of ' + IntToStr(AllFoundList.Count) + RUNNING);
                  Progress.Log(StringOfChar('-', CHAR_LENGTH));
                end;

                // Set the iTunes Domain String
                iTunes_Domain_str := '';
                if assigned(FieldItunesDomain) then
                begin
                  try
                    iTunes_Domain_str := FieldItunesDomain.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Domain string');
                  end;
                end;

                // Set the iTunes Name String
                iTunes_Name_str := '';
                if assigned(FieldItunesName) then
                begin
                  try
                    iTunes_Name_str := FieldItunesName.AsString[anEntry];
                  except
                    Progress.Log(ATRY_EXCEPT_STR + 'Error reading iTunes Name string');
                  end;
                end;

                // Run the match
                aDeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
                if
                (anEntry.Extension <> 'db-shm') and
                (anEntry.Extension <> 'db-wal') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite WAL') and
                (aDeterminedFileDriverInfo.ShortDisplayName <> 'Sqlite SHM') then
                begin
                  if RegexMatch(anEntry.EntryName, regex_search_str, False) or
                  RegexMatch(anEntry.FullPathName, regex_search_str, False) or
                  FileSubSignatureMatch(anEntry) or
                  RegexMatch(iTunes_Domain_str, regex_search_str, False) or
                  RegexMatch(iTunes_Name_str, regex_search_str, False) then
                  begin

                    // Sub Signature Match
                    if FileSubSignatureMatch(anEntry) then
                    begin
                      if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag2]; // Blue Flag
                      DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                    end
                    else

                    // Regex Name Match
                    begin
                      if ((not anEntry.isDirectory) or (anEntry.isDevice)) and (anEntry.LogicalSize > 0) and (anEntry.PhysicalSize > 0) then
                      begin
                        if FileNameRegexSearch(anEntry, iTunes_Domain_str, iTunes_Name_str, regex_search_str) then
                        begin
                          if BL_USE_FLAGS then anEntry.Flags := anEntry.Flags + [Flag1]; // Red Flag
                          DetermineThenSkipOrAdd(anEntry, iTunes_Domain_str, iTunes_Name_str);
                        end;
                      end;
                    end;

                  end;
                  Progress.IncCurrentprogress;
                end;
              end;
            end;
          finally
            AllFoundList.free;
          end;

          // Check to see if files were found
          if (TotalValidatedFileCountInTLists = 0) then
          begin
            Progress.Log('No ' + PROGRAM_NAME + SPACE + 'files were found.');
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Not found';
          end
          else
          begin
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
            Progress.Log(RPad('Total Validated Files:', RPAD_VALUE) + IntToStr(TotalValidatedFileCountInTLists));
            Progress.Log(StringOfChar('=', CHAR_LENGTH + 80));
          end;

          // Display the content of the TLists for further processing
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.Log('Lists available to process' + RUNNING);
            for n := 0 to Length(gArr) - 1 do
            begin
              if not Progress.isRunning then
                break;
              Item := gArr[n];
              if gArr_ValidatedFiles_TList[n].Count > 0 then
              begin
                Progress.Log(StringOfChar('-', CHAR_LENGTH));
                Progress.Log(RPad('TList ' + IntToStr(n) + ': ' + Item.fi_Name_Program + SPACE + Item.fi_Name_Program_Type + HYPHEN + Item.fi_Name_OS, RPAD_VALUE) + IntToStr(gArr_ValidatedFiles_TList[n].Count));
                for r := 0 to gArr_ValidatedFiles_TList[n].Count - 1 do
                begin
                  if not Progress.isRunning then
                    break;
                  anEntry := (TEntry(TList(gArr_ValidatedFiles_TList[n]).items[r]));
                  if not(Item.fi_Process_As = 'POSTPROCESS') and BL_PROCEED_LOGGING then
                    Progress.Log(RPad(' Bates: ' + IntToStr(anEntry.ID), RPAD_VALUE) + anEntry.EntryName + SPACE + Item.fi_Process_As);
                end;
              end;
            end;
            Progress.Log(StringOfChar('-', CHAR_LENGTH + 80));
          end;

          // *** CREATE GUI ***
          if (TotalValidatedFileCountInTLists > 0) and Progress.isRunning then
          begin
            Progress.DisplayTitle := 'Artifacts' + HYPHEN + PROGRAM_NAME + HYPHEN + 'Found';

            Progress.Log(RPad('Process All Lists' + RUNNING, RPAD_VALUE) + IntToStr(High(gArr) + 1)); // noslz
            Progress.Log(StringOfChar('=', CHAR_LENGTH));

            // *** DO PROCESS ***
            temp_process_counter := 0;
            Progress.CurrentPosition := 0;
            Progress.Max := TotalValidatedFileCountInTLists;
            gtick_doprocess_i64 := GetTickCount;

            // *****************************************************************
            for n := 0 to Length(gArr) - 1 do
            begin
              ref_num := n;
              if (TestForDoProcess(ref_num)) then
              begin
                Item := gArr[n];
                Process_ID_str := Item.fi_Process_ID;
                DoProcess(gArtConnect_ProgFldr[ref_num], ref_num, Location_Columns.GetTable(Process_ID_str));
              end;
            end;
            // *****************************************************************

            gtick_doprocess_str := (RPad('DoProcess:', RPAD_VALUE) + CalcTimeTaken(gtick_doprocess_i64));

          end;

          if not Progress.isRunning then
          begin
            Progress.Log(CANCELED_BY_USER);
            Progress.DisplayMessageNow := CANCELED_BY_USER;
          end;

        finally
          for n := 0 to Length(gArr) - 1 do
            FreeAndNil(gArr_ValidatedFiles_TList[n]);
        end;

      finally
        if assigned(gFileSystemDataStore) then
          FreeAndNil(gFileSystemDataStore);
      end;

    finally
      if assigned(gArtifactsDataStore) then
        FreeAndNil(gArtifactsDataStore);
    end;

  finally
    if assigned(gParameter_Num_StringList) then
      FreeAndNil(gParameter_Num_StringList);
  end;

  if Progress.isRunning then
  begin
    Progress.Log(gtick_foundlist_str);
    Progress.Log(gtick_doprocess_str);
  end;

  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  Progress.Log(SCRIPT_NAME + ' finished.');
  Progress.DisplayMessageNow := ('Processing complete.');

end.
