unit WeChat_Decrypt_EnMicroMsg;

interface

uses
{$IF DEFINED (ISFEXGUI)}
  GUI, Modules,
{$IFEND}
  Classes, Common, Contnrs, DataEntry, DataStorage, Graphics,
  RegEx, SysUtils, XML;

const
  // WeChat Identifiers
  GIMEI_STR = '1234567890ABCDEF'; //  // noslz Default value for Android 10+
  WECHAT_DB = 'EnMicroMsg.db'; //  // noslz Database name

  BS = '\';
  CHAR_LENGTH = 80;
  CR = #13#10;
  DCR = #13#10 + #13#10;
  HYPHEN = ' - ';
  OSFILENAME = '\\?\';
  RPAD_VALUE = 35;
  RUNNING = '...';
  SCRIPT_NAME = 'WeChat Decrypt EnMicroMsg';
  SPACE = ' ';
  TSWT = 'The script will terminate.';
  FAILED_STR = 'WeChat decryption failed.';
  FORMAT_STR = '%-25s %-15s %-15s %-15s %-15s';
  SPECIALCHAR_TICK: char = #10004; // noslz
  SPECIALCHAR_X: char = #10006; // noslz

var
  gDataStore_FS: TDataStore;
  gGlob_SysConfig_StringList: TStringList;
  gGlob_WeChatDB_StirngList: TStringList;
  gSysConfig_TList: TList;
  gUIN_str: string;
  gUIN_StringList: TStringList;
  gWeChatDBs_TList: TList;

implementation

// -----------------------------------------------------------------------------
// Global Search - system_config_prefs.xml
// -----------------------------------------------------------------------------
procedure GlobalSearch_SysConfig(aDataStore: TDataStore; aTList: TList; aStringList: TStringList);
var
  i: integer;
begin
  if assigned(aDataStore) and (aDataStore.Count > 1) then
  begin
    for i := 0 to aStringList.Count - 1 do
      aDataStore.FindEntriesByPath(nil, aStringList[i], aTList);
  end;
end;

// -----------------------------------------------------------------------------
// Global Search - Find Files by Path
// -----------------------------------------------------------------------------
procedure GlobalSearch(aDataStore: TDataStore; aTList: TList; aStringList: TStringList);
var
  aEntry: TEntry;
  dEntry: TEntry;
  fEntry: TEntry;

  device_str: string;
  Enum: TEntryEnumerator;
  Found_TList: TList;
  i: integer;
  iIdx: integer;
  Unique_List: TUniqueListOfEntries;

begin
  Progress.Log(format(FORMAT_STR, ['Files by Path', 'Bates#', 'Size (bytes)', 'Decrypted', 'Device']));
  Progress.Log(format(FORMAT_STR, ['-------------', '------', '------------', '---------', '------']));

  if assigned(aDataStore) and (aDataStore.Count > 1) then
  begin
    Found_TList := TList.Create;
    Unique_List := TUniqueListOfEntries.Create;
    try
      for i := 0 to aStringList.Count - 1 do
      begin
        aDataStore.FindEntriesByPath(nil, aStringList[i], Found_TList);
        for iIdx := 0 to Found_TList.Count - 1 do
        begin
          fEntry := TEntry(Found_TList[iIdx]);
          dEntry := GetDeviceEntry(fEntry);
          if assigned(dEntry) then
            device_str := dEntry.EntryName;

          if not fEntry.isEncrypted then
            Unique_List.Add(fEntry)
          else
            Progress.Log(format(FORMAT_STR, [HYPHEN + fEntry.EntryName, inttostr(fEntry.ID), IntToStr(fEntry.LogicalSize), BoolToStr(fEntry.isEncrypted, True) + SPACE + SPECIALCHAR_TICK, device_str]))
        end;
        Found_TList.Clear;
      end;
      // Put the Unique List into the received TList
      Enum := Unique_List.GetEnumerator;
      while Enum.MoveNext do
      begin
        aEntry := Enum.Current;
        if RegexMatch(aEntry.Extension, '.(DB|DB-WAL)$', False) then
        begin
          aTList.Add(aEntry);
          dEntry := GetDeviceEntry(aEntry);
          if assigned(dEntry) then
            device_str := dEntry.EntryName;
          Progress.Log(format(FORMAT_STR, [HYPHEN + aEntry.EntryName, inttostr(aEntry.ID), IntToStr(aEntry.LogicalSize), BoolToStr(aEntry.isEncrypted, True) + SPACE + SPECIALCHAR_X, device_str]))
        end;
      end;
    finally
      Found_TList.free;
      Unique_List.free;
    end;
  end;
  Progress.Log(StringOfChar('-', CHAR_LENGTH));
end;

// -----------------------------------------------------------------------------
// Main Proc
// -----------------------------------------------------------------------------
procedure MainProc;
var
  i: integer;

begin
  gGlob_WeChatDB_StirngList.Add('**\' + WECHAT_DB + '*'); // noslz - Collect both db and wal
  GlobalSearch(gDataStore_FS, gWeChatDBs_TList, gGlob_WeChatDB_StirngList);

  if gWeChatDBs_TList.Count <= 0 then
  begin
    Progress.Log('No encrypted files found.' + SPACE + TSWT);
    Exit;
  end;

  // Decrypt for each UIN
  for i := 0 to gUIN_StringList.Count - 1 do
  begin
    Progress.Log('Decryption Run:' + SPACE + IntToStr(i+1));
    gUIN_str := Trim(gUIN_StringList[i]);
    if Decrypt_WeChat_TList then
    begin
      SignatureAnalysis(gWeChatDBs_TList); // Get the signature after the decryption
      Progress.Log('WeChat processing completed successfully.')
    end
    else
    begin
      Progress.Log(FAILED_STR + SPACE + TSWT);
      Exit;
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Message User
// -----------------------------------------------------------------------------
procedure MessageUser(aString: string);
begin
  Progress.Log(aString);
{$IF DEFINED (ISFEXGUI)}
  MessageBox(aString, SCRIPT_NAME, (MB_OK or MB_ICONINFORMATION or MB_SETFOREGROUND or MB_TOPMOST));
{$IFEND}
end;

// -----------------------------------------------------------------------------
// Process WeChat (Decrypt)
// -----------------------------------------------------------------------------
function Decrypt_WeChat_TList: boolean;
var
  TaskProgress: TPAC;
  Parameters: array of const;
begin
  Result := False;
  if (gWeChatDBs_TList.Count > 0) then
  begin
    Parameters := ['imei=' + GIMEI_STR, 'uin=' + gUIN_str]; // noslz
    try
      TaskProgress := Progress.NewChild;
      RunTask('TCommandTask_WeChat', DATASTORE_FILESYSTEM, gWeChatDBs_TList, TaskProgress, Parameters);
      while (TaskProgress.isRunning) and (Progress.isRunning) do
        Sleep(500);
      Result := (TaskProgress.CompleteState = pcsSuccess);
      if TaskProgress.CompleteState = pcsFailed then
        Result := False;
    except
      on e: exception do
        Progress.Log(e.message);
    end;
  end;
end;

// -----------------------------------------------------------------------------
// Process XML (Extract UIN)
// -----------------------------------------------------------------------------
procedure ProcessXML(anEntry: TEntry; aStringList: TStringList);
const
  search_str = '<int name="default_uin" value="'; // noslz
var
  EntryReader: TEntryReader;
  i: integer;
  outStringList: TStringList;
  tmp_str: string;
begin
  outStringList:= TStringList.Create;
  EntryReader := TEntryReader.Create;
  try
    if (anEntry.LogicalSize > 16) and (EntryReader.OpenData(anEntry)) then
    begin
      EntryReader.Position := 0;
      outStringList.LoadFromStream(EntryReader);
      if outStringList.Count > 0 then
      begin
        for i := 0 to outStringList.Count - 1 do
        begin
          if Pos(search_str, outStringList[i]) > 0 then
          begin
            tmp_str := '';
            tmp_str := StringReplace(outStringList[i], search_str, '', []);
            tmp_str := StringReplace(tmp_str, '" />', '', []);
            Trim(tmp_str);
            if RegexMatch(tmp_str, '\d+', True) then // noslz
            begin
              aStringList.Add(Trim(tmp_str));
              Progress.Log(format(FORMAT_STR, [HYPHEN + 'Found UIN:', IntToStr(anEntry.ID), Trim(tmp_str), '', '']));
            end;
          end;
        end;
      end;
    end;
  finally
    outStringList.free;
    EntryReader.free;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString) + 1;
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

// -----------------------------------------------------------------------------
// Signature Analysis (TCommandTask_FileTypeAnalysis)
// -----------------------------------------------------------------------------
function SignatureAnalysis(aTList: TList): boolean;
var
  TaskProgress: TPAC;
begin
  Result := False;
  if (not Progress.isRunning) or (not assigned(aTList)) then
    Exit;
  if aTList.Count > 0 then
  begin
    Progress.Log('Signature Analysis:' + SPACE + IntToStr(aTList.Count));
    Progress.DisplayMessageNow := 'Launching Signature Analysis (' + IntToStr(aTList.Count) + ' files)' + RUNNING;
    TaskProgress := Progress.NewChild; // NewProgress(True);
    try
      RunTask('TCommandTask_FileTypeAnalysis', DATASTORE_FILESYSTEM, aTList, TaskProgress, ['forcedetermination=True']); // noslz
    except
      on e: exception do
        Progress.Log(e.message);
    end;
    while (TaskProgress.isRunning) and (Progress.isRunning) do
      Sleep(500); // Do not proceed until TCommandTask is complete
    if not(Progress.isRunning) then
      TaskProgress.Cancel;
    Result := (TaskProgress.CompleteState = pcsSuccess);
    Progress.Log(StringOfChar('-', CHAR_LENGTH));
    Progress.DisplayTitle := SCRIPT_NAME;
  end;
end;

// -----------------------------------------------------------------------------
// Starting Checks
// -----------------------------------------------------------------------------
function StartingChecks: boolean;
var
  StartCheckDataStore: TDataStore;
  StartCheckEntry: TEntry;
begin
  Result := False;
  Progress.DisplayTitle := SCRIPT_NAME;
  StartCheckDataStore := GetDataStore(DATASTORE_FILESYSTEM);
  if not assigned(StartCheckDataStore) then
  begin
    Progress.Log(DATASTORE_FILESYSTEM + ' module not located.' + SPACE + TSWT);
    Exit;
  end;
  try
    StartCheckEntry := StartCheckDataStore.First;
    if not assigned(StartCheckEntry) then
    begin
      Progress.Log('There is no current case.' + SPACE + TSWT);
      Exit;
    end;
    if StartCheckDataStore.Count <= 1 then
    begin
      Progress.Log('There are no files in the File System module.' + SPACE + TSWT);
      Exit;
    end;
    Result := True;
  finally
    StartCheckDataStore.free;
  end;
end;

// =============================================================================
// Start of Script
// =============================================================================
const
  SCRIPT_FINISHED_STR = 'Script finished.';

var
  scEntry: TEntry;
  i: integer;

begin
  Progress.Log('Starting' + RUNNING);
  Progress.Log(StringOfChar('-', CHAR_LENGTH));

  if not StartingChecks then
  begin
    Progress.DisplayMessageNow := (SCRIPT_FINISHED_STR);
    Exit;
  end;

  gDataStore_FS := GetDataStore(DATASTORE_FILESYSTEM);
  if gDataStore_FS = nil then
    Exit;

  try
    gSysConfig_TList := TList.Create;
    gUIN_StringList := TStringList.Create;
    gUIN_StringList.Sorted := True;
    gUIN_StringList.Duplicates := dupIgnore;
    gWeChatDBs_TList := TList.Create;
    gGlob_SysConfig_StringList := TStringList.Create;
    gGlob_WeChatDB_StirngList := TStringList.Create;
    try
      gGlob_SysConfig_StringList.Add('**\system_config_prefs.xml*'); // noslz
      GlobalSearch_SysConfig(gDataStore_FS, gSysConfig_TList, gGlob_SysConfig_StringList);

      if gSysConfig_TList.Count = 0 then
      begin
        MessageUser('WeChat' + ':' + DCR + 'Did not locate a system_config_prefs.xml file.' + DCR + TSWT);
        Progress.Log(SCRIPT_FINISHED_STR);
        Exit;
      end;

      // Collect the UIN from system_config_prefs.xml
      Progress.Log(format(FORMAT_STR, ['Search for UIN' + RUNNING, 'Bates#', 'UIN', '', '']));
      Progress.Log(format(FORMAT_STR, ['--------------:',          '------', '---', '', '']));
      for i := 0 to gSysConfig_TList.Count - 1 do
      begin
        scEntry := TEntry(gSysConfig_TList[i]);
        ProcessXML(scEntry, gUIN_StringList);
      end;

      if gUIN_StringList.Count = 0 then
      begin
        MessageUser('No UIN found.' + SPACE + TSWT);
        Progress.Log(StringOfChar('-', CHAR_LENGTH));
        Exit;
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      Progress.Log('Unique UIN:' + IntToStr(gUIN_StringList.Count));
      if gUIN_StringList.Count > 0 then
      begin
        for i := 0 to gUIN_StringList.Count - 1 do
          Progress.Log(HYPHEN + gUIN_StringList[i]);
      end;
      Progress.Log(StringOfChar('-', CHAR_LENGTH));

      MainProc;

    finally
      gGlob_SysConfig_StringList.free;
      gGlob_WeChatDB_StirngList.free;
      gSysConfig_TList.free;
      gUIN_StringList.free;
      gWeChatDBs_TList.free;
    end;

  finally
    gDataStore_FS.free;
  end;

  Progress.Log(SCRIPT_FINISHED_STR);
  Progress.DisplayMessageNow := (SCRIPT_FINISHED_STR);

end.
