﻿unit Encrypted_Files;

interface

uses
  Classes, Clipbrd, Common, DataEntry, DataStorage, Graphics, Math, PropertyList, Regex, SysUtils;

const
  MSENCRYPTED           = 'MS ENCRYPTED'; //noslz - set for UpperCase
  AUTHOR                = 'GetData';
  BM_SCRIPT_OUTPUT_FLDR = 'FEX-Triage';
  BM_ENCRYPTED_FILES    = 'Encrypted Files';
  BS                    = '\';
  CHAR_LENGTH           = 80;
  COMMA                 = ', ';
  HYPHEN                = ' - ';
  RPAD_VALUE            = 30;
  RUNNING               = '...';
  SCRIPT_NAME           = 'Find Encrypted Files';
  SCRIPT_DESCRIPTION    = 'Find encrypted files.' + #13#10 +
                          'Files located can be bookmarked from the results screen.' + #13#10 + #13#10 +
                          'WARNING: This is not an exhaustive search for all encryption types.';
  SCRIPT_REFERENCES     = 'Entropy: http://en.wikipedia.org/wiki/Entropy_%28information_theory%29' + #13#10 +
                          'RAR v5 Archive Format: https://www.rarlab.com/technote.htm#enchead';
  SPACE                 = ' ';
  STARTUP_TAB           = 1;
  BUFFERSIZE            = 4096*4;
  TSWT                  = 'The script will terminate.';

var
  filebuffer: array[0..16383] of byte; // Must be global
  gTotal_Count: integer;
  bl_Continue: boolean;

  // Global StringLists
  isEncryptedTest_StringList: TStringList;
  HighEntropy_StringList: TStringList;
  Memo_StringList: TStringList;
  NotDetected_StringList: TStringList;
  RAR5Test_StringList: TStringList;
  ZipMetadataTest_StringList: TStringList;

  // Global TLists
  Encrypted_TUniqueList: TUniqueListOfEntries;
  HighEntropy_TList: TList;
  FieldEntropy: TDataStoreField;
  Enum: TEntryEnumerator;

function ListExists(aList: TListOfEntries; ListName_str: string) : boolean;
function ParametersReceived : boolean;
function RPad(const AString: string; AChars: integer): string;
procedure AddTo_Audit_Bucket(aEntry: TEntry; aStringList: TStringList; aname_str: string);
procedure AddToEncryptedInExpanded(eEntry: TEntry; Encrypted_TUniqueList: TList);
procedure Bookmark (abmFolder: string; aTList: TList);
procedure CreatePDF(aStringList: TStringList; FileSaveName_str: string; font_size: integer);
procedure DetermineSig(anEntry: TEntry);
procedure DetectEncryptedPartitions;
procedure MainProc;
procedure ProgressLogMemo(AString: string);

implementation

function ShellExecute(hWnd: cardinal; lpOperation: Pchar; lpFile: Pchar; lpParameter: Pchar; lpDirectory: Pchar; nShowCmd: integer): Thandle; stdcall; external 'Shell32.Dll' name 'ShellExecuteW';

//------------------------------------------------------------------------------
// Function: Right Pad v2
//------------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString);
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

//------------------------------------------------------------------------------
// Function: Make sure the InList and OutList exist
//------------------------------------------------------------------------------
function ListExists(aList: TListOfEntries; ListName_str: string) : boolean;
begin
  Result := False;
  if assigned(aList) then
  begin
    Progress.Log(RPad(ListName_str + SPACE + 'count:', rpad_value) + IntToStr(aList.Count));
    Result := True;
  end
  else
    Progress.Log(RPad('Not Assigned:', rpad_value) + ListName_str);
end;

//------------------------------------------------------------------------------
// Function: Parameters Received
//------------------------------------------------------------------------------
function ParametersReceived : boolean;
var
  param_str: string;
  i: integer;
begin
  Result := False;
  Progress.Log(Stringofchar('-', 80));
  if CmdLine.ParamCount > 0 then
  begin
    Result := True;
    param_str := '';
    Progress.Log(RPad('Params received:', RPAD_VALUE) + IntToStr(CmdLine.ParamCount));
    Progress.Log('Params:');
    for i := 0 to CmdLine.ParamCount - 1 do
    begin
      Progress.Log(RPad(' Param ' + IntToStr(i) + ':', RPAD_VALUE) + cmdline.params[i]);
      param_str := param_str + '"' + cmdline.params[i] + '"' + ' ';
    end;
    trim(param_str);
  end
  else
    Progress.Log('No parameters were received.');
  Progress.Log(Stringofchar('-', 80));
end;

//------------------------------------------------------------------------------
// Procedure: Progress.Log to Memo_StringList
//------------------------------------------------------------------------------
procedure ProgressLogMemo(AString: string);
begin
  Progress.Log(AString);
  if assigned(Memo_StringList) and (Memo_StringList.Count < 5000) then
    Memo_StringList.Add(AString);
end;

//------------------------------------------------------------------------------
// Procedure: Find the parent of an Encrypted in Expanded
//------------------------------------------------------------------------------
procedure AddToEncryptedInExpanded(eEntry: TEntry; Encrypted_TUniqueList: TList);
var
  e,f: integer;
  InList_Entry: TEntry;
  testEntry: TEntry;
  upper_shortdisplayname_str: string;
begin
  if assigned(Encrypted_TUniqueList) then
  begin
    if eEntry.isEncrypted and eEntry.inExpanded then
    begin
      testEntry := TEntry(eEntry.parent);
      for f := 0 to 50 do
      begin
        if not testEntry.isExpanded then
          testEntry := TEntry(testEntry.parent)
        else
          break;
      end;
      for e := 0 to Encrypted_TUniqueList.Count - 1 do
      begin
        InList_Entry := TEntry(Encrypted_TUniqueList[e]);
        if testEntry = InList_Entry then
        Exit;
      end;

      upper_shortdisplayname_str := UpperCase(testEntry.FileDriverInfo.ShortDisplayname);

      if ((upper_shortdisplayname_str = 'XLSX') or
         (upper_shortdisplayname_str = 'DOCX') or
         (upper_shortdisplayname_str = 'PPTX') or
         (upper_shortdisplayname_str <> MSENCRYPTED)) then
      begin
        Encrypted_TUniqueList.Add(testEntry);
        bl_Continue := False;
        Exit;
      end;

      if ((upper_shortdisplayname_str <> 'XLSX') and
         (upper_shortdisplayname_str <> 'DOCX') and
         (upper_shortdisplayname_str <> 'PPTX') and
         (upper_shortdisplayname_str <> MSENCRYPTED)) then
      begin
        Encrypted_TUniqueList.Add(testEntry);
        bl_Continue := False;
        Exit;
      end;

    end;
  end;
end;

//------------------------------------------------------------------------------
// Procedure: Bookmark
//------------------------------------------------------------------------------
procedure Bookmark (abmFolder: string; aTList: TList);
var
  bmfolder: TEntry;
  bmpath: string;
  bmCreate: boolean;
  bmEntry: TEntry;
  k: integer;
begin
  bmPath  := BM_SCRIPT_OUTPUT_FLDR + BS + BM_ENCRYPTED_FILES + BS + abmFolder;
  bmCreate := True;
  bmFolder := FindBookmarkByName(bmPath, bmCreate);
  if assigned(bmFolder) and (aTList.Count > 0) then
  for k := 0 to (aTList.Count - 1) do
  begin
    bmEntry := TEntry(aTList[k]);
    if not IsItemInBookmark(bmFolder, bmEntry) then
    begin
      AddItemsToBookmark(bmFolder, DATASTORE_FILESYSTEM, bmEntry, '');
    end;
  end;
end;

//------------------------------------------------------------------------------
// Function: Calculate Entropy
//------------------------------------------------------------------------------
function CalculateEntropy(calcEntry: TEntry) : extended;
var
  b: integer;
  byteCounts: array [0 .. 255] of int64;
  bytesRead: integer;
  calcEntropy: extended;
  p : double;
  ReadStream: Boolean;
  calcReader: TEntryReader;
begin
  if (calcEntry.LogicalSize > 0) and (calcEntry.PhysicalSize > 0) then
  begin
    calcReader := TEntryReader.Create;
    calcEntropy := 0.0;
    if calcReader.OpenData(calcEntry) then
    begin
      for b := 0 to 255 do
        byteCounts[b] := 0;

      try
        bytesRead := calcReader.Read(filebuffer[0],BufferSize);
        if bytesRead > 0 then
        begin
          for b := 0 to bytesRead-1 do
            Inc(byteCounts[filebuffer[b]]);
          for b := 0 to 255 do
          begin
             p := byteCounts[b]/bytesRead;
             if p > 0.0  then
               calcEntropy := calcEntropy - p*Log2(p);
          end;
          calcEntropy := calcEntropy*0.125;
        end;
      except
        ReadStream := False;
      end;
      FieldEntropy.AsExtended[calcEntry] := calcEntropy;
    end
    else
      Progress.Log('Could not open data: ' + calcEntry.EntryName);
  end;
  Result := calcEntropy;
  calcReader.free;
end;

//------------------------------------------------------------------------------
// Procedure: Audit bucket is displayed in the messages window
//------------------------------------------------------------------------------
procedure AddTo_Audit_Bucket(aEntry: TEntry; aStringList: TStringList; aname_str: string);
var
  Entropy_str: string;
  fileext: string;
begin
  fileext := UpperCase(aEntry.EntryNameExt);
  if assigned(aStringList) then
  begin
    if aEntry.IsDirectory and not aEntry.IsExpanded then
      Entropy_str := 'Folder'
    else
      Entropy_str := FieldEntropy.asString[aEntry];
    aStringList.add(format('%-10s %-10s %-25s %-100s',[SPACE + IntToStr(aEntry.ID), Entropy_str, aname_str, aEntry.FullPathName]));
  end;
end;

//------------------------------------------------------------------------------
// Procedure: Determine Signature
//------------------------------------------------------------------------------
procedure DetermineSig(anEntry: TEntry);
var
  Report_Time_Taken_bl: boolean;
  Starting_Tick_Count: dword;
  Time_Taken_int: integer;
begin
  Report_Time_Taken_bl := False;
  starting_tick_count := 0;
  starting_tick_count := GetTickCount;
  DetermineFileType(anEntry);
  if Report_Time_Taken_bl then
  begin
    time_taken_int := (GetTickCount - starting_tick_count);
    if time_taken_int > 100 then
      Progress.Log(RPad(anEntry.EntryName + ': ', 40) + 'Determine Signature Time: ' + IntToStr(time_taken_int));
  end;
end;

//------------------------------------------------------------------------------
// Procedure: Create PDF
//------------------------------------------------------------------------------
procedure CreatePDF(aStringList: TStringList; FileSaveName_str: string; font_size: integer);
var
  CurrentCaseDir: string;
  SaveDir: string;
begin
  CurrentCaseDir := GetCurrentCaseDir;
  SaveDir := CurrentCaseDir + 'Reports\Run Time Reports\';
  if assigned(aStringList) and (aStringList.Count > 0) then
  begin
    if ForceDirectories(IncludeTrailingPathDelimiter(SaveDir)) and DirectoryExists(SaveDir) then
    begin
      if GeneratePDF(SaveDir + FileSaveName_str, aStringList, font_size) then
        Progress.Log('Saved: ' + SaveDir + FileSaveName_str)
      else
        Progress.Log('Failed to save: ' + SaveDir + FileSaveName_str);
    end
    else
      Progress.Log('Could not find folder: ' + SaveDir);
  end;
end;

//------------------------------------------------------------------------------
// Procedure: MainProc
//------------------------------------------------------------------------------
procedure MainProc;
var
  anEntry: TEntry;
  anExpanded7zList: TList;
  aParentNode: TPropertyNode;
  aPropertyTree: TPropertyParent;
  aReader: TEntryReader;
  aRootProperty: TPropertyNode;
  ByteArray: array[1..8] of byte;
  ByteCount: integer;
  ChildEntry: TEntry;
  DeterminedFileDriverInfo: TFileTypeInformation;
  DeterminedSig: string;
  encEntry: TEntry;
  Entropy_str: string;
  EntryList: TDataStore;
  fileEntropy: extended;
  fileext: string;
  FolderChildren_List: TList;
  Found_Count_str: string;
  hexbytes: string;
  i, s, w: integer;
  iTunesBackup_Children_TList: TList;
  Heading_str: string;
  Starting_Byte: integer;
  TaskProgress: TPAC;
  Temp_TList: TList;
  upper_shortdisplayname_str: string;
  Warning_PDF_StringList: TStringList;

  // Starting Checks
  CaseName:                   string;
  FEX_Executable:             string;
  FEX_Version:                string;
  PAS_File_Name:              string;
  ScriptPath_str:             string;

  UseInList_bl: boolean;
  Working_TList: TList;
  file_ext: string;

begin
  EntryList := GetDataStore(DATASTORE_FILESYSTEM);
  if not assigned(EntryList) then
  begin
    Progress.Log('Not assigned datastore. The script will terminate.');
    Exit;
  end;
  anEntry := EntryList.First;
  if not assigned(anEntry) then
  begin
    EntryList.free;
    Progress.Log('No current case. The script will terminate.');
    Exit;
  end
  else
    CaseName := anEntry.EntryName;

  ScriptPath_str := CmdLine.Path;
  PAS_File_Name := ExtractFileName(ScriptPath_str);
  FEX_Executable := GetApplicationDir + 'forensicexplorer.exe';
  FEX_Version := GetFileVersion(FEX_Executable);

  Memo_StringList := TStringList.Create;
  ProgressLogMemo(RPad('Case Name:', RPAD_VALUE) + CaseName);
  ProgressLogMemo(RPad('Script Name:', RPAD_VALUE) + SCRIPT_NAME);
  ProgressLogMemo(RPad('Script File:', RPAD_VALUE) + PAS_File_Name);
  ProgressLogMemo(RPad('Date Run:', RPAD_VALUE) + DateTimeToStr(now));
  ProgressLogMemo(RPad('FEX Version:', RPAD_VALUE) + FEX_Version);

  // Progress
  Progress.Log(SCRIPT_NAME + ' starting' + RUNNING);
  Progress.Initialize(EntryList.Count, SCRIPT_NAME + ' starting' + RUNNING);
  Progress.CurrentPosition := 1;
  Progress.Max := EntryList.Count;

  // Create StringLists
  isEncryptedTest_StringList := TStringList.Create;
  isEncryptedTest_StringList.Sorted := True; // Must be sorted or duplicates in the following line has no effect
  isEncryptedTest_StringList.Duplicates := dupIgnore;
  HighEntropy_StringList := TStringList.Create;
  Memo_StringList := TStringList.Create;
  NotDetected_StringList := TStringList.Create;
  RAR5Test_StringList := TStringList.Create;
  ZipMetadataTest_StringList := TStringList.Create;

  // Create TLists
  Encrypted_TUniqueList := TUniqueListOfEntries.Create;
  FolderChildren_List := TList.Create;
  HighEntropy_TList := TList.Create;

  //DetectEncryptedPartitions;

  if (CmdLine.Params.Indexof('InList') >= 0) then
    UseInList_bl := True
  else
    UseInList_bl := False;

  Working_TList := TList.Create;
  try

     UseInList_bl := False;
    //-------------------------------------------------------------------------------
    // Do All
    //-------------------------------------------------------------------------------
    if UseInList_bl = False then
    begin
      Working_TList := EntryList.GetEntireList;
      Progress.Log(RPad('Working with All items:', rpad_value) + IntToStr(Working_TList.Count));
    end
    //-------------------------------------------------------------------------------
    // Else Use InList and OutList
    //-------------------------------------------------------------------------------
    else
    begin
      Progress.Log('Working with InList:');
      if (not ListExists(InList, 'InList')) or (not ListExists(OutList, 'OutList')) then
        Exit
      else
      begin
        if InList.Count = 0 then
        begin
          Progress.Log('InList count is zero.' + SPACE + TSWT);
          Exit;
        end;
      end;
      for i := 0 to InList.Count -1 do
      begin
        anEntry := TEntry(InList[i]);
        Working_TList.Add(anEntry);
      end;
    end;

    FieldEntropy := AddMetaDataField('Entropy', ftExtended);
    aReader := TEntryReader.Create;

    try
      Found_Count_str := '';
      Progress.DisplayMessageNow := ('Searching for encrypted files' + RUNNING);

      Progress.Log(RPad('Working_TList Count:', rpad_value) + IntToStr(Working_TList.Count));
      for w := 0 to Working_TList.Count -1 do
      begin
	    if not Progress.isRunning then break;
        anEntry := TEntry(Working_TList[w]);
        bl_Continue := True;

        if anEntry.PhysicalSize < 500 then
        begin
          Progress.IncCurrentprogress;
          continue;
        end;

        file_ext := UpperCase(anEntry.EntryNameExt);
        if (file_ext = '.ACCDT') or
          (file_ext = '.CAB') or
          (file_ext = '.EFTX') or
          (file_ext = '.GLOX') or
          (file_ext = '.GZ') or
          (file_ext = '.IPA') or
          (file_ext = '.JAR') or
          (file_ext = '.MSHC') or
          (file_ext = '.THMX') or
          (file_ext = '.XAP') then
        begin
          Progress.IncCurrentprogress;
          continue;
          Progress.Log('Skipped: ' + anEntry.EntryName);
        end;

        //------------------------------------------------------------------
        // Check for "Encrypted" iTunes Backups - Manifest Plist 
        //------------------------------------------------------------------
        if bl_Continue then
        begin
          if anEntry.EntryName = 'Manifest.plist' then
          begin
            DetermineSig(anEntry);
            DeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
            DeterminedSig := DeterminedFileDriverInfo.ShortDisplayName;
            if aReader.OpenData(anEntry) then
            begin
              aPropertyTree := nil;
              if ProcessMetadataProperties(anEntry, aPropertyTree, aReader) then
              try
                aRootProperty := aPropertyTree;
                if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                begin
                  aParentNode := TPropertyNode(aPropertyTree.GetFirstNodeByName('isEncrypted')); //noslz
                  if assigned(aParentNode) then
                  begin
                    if aParentNode.PropValue = True then
                    begin
                      Encrypted_TUniqueList.Add(anEntry.ParentEntry);
                      CalculateEntropy(anEntry);
                      if (UpperCase(DeterminedSig) = 'PLIST (BINARY)') then
                      begin
                        AddTo_Audit_Bucket(anEntry.ParentEntry, isEncryptedTest_StringList, 'iTunes');
                        iTunesBackup_Children_TList := EntryList.Children(anEntry.ParentEntry);
                        Temp_TList := TList.Create;
                        try
                          for s := 0 to iTunesBackup_Children_TList.Count - 1 do
                          begin
                            ChildEntry := TEntry(iTunesBackup_Children_TList[s]);
                            if UpperCase(ChildEntry.EntryName) = 'MANIFEST.MBDB' then
                            begin
                              Temp_TList.Add(ChildEntry);
                            end;
                          end;

                          if assigned(Temp_TList) and (Temp_TList.Count > 0) then
                          begin
                            //TaskProgress := NewProgress(True);
                            TaskProgress := Progress.NewChild;
                            RunTask('TCommandTask_AppleBackup', DATASTORE_FILESYSTEM, Temp_TList, TaskProgress);
                            while Not TaskProgress.IsFinished do
                            begin
                              Sleep(500);
                            end;
                            Progress.DisplayTitle := SCRIPT_NAME;  // Refresh the title
                            TaskProgress.DisplayMessages := ('Searching for encrypted files (Found: ' + Found_Count_str + ')' + RUNNING);
                            Progress.DisplayMessages := ('Searching for encrypted files (Found: ' + Found_Count_str + ')' + RUNNING);

                            for s := 0 to iTunesBackup_Children_TList.Count - 1 do
                            begin
                              ChildEntry := TEntry(iTunesBackup_Children_TList[s]);
                              if ChildEntry.isEncrypted then
                              begin
                                Encrypted_TUniqueList.Add(ChildEntry);
                                CalculateEntropy(ChildEntry);
                                AddTo_Audit_Bucket(ChildEntry, isEncryptedTest_StringList, 'isEncrypted Other');
                                //bl_Continue := False;
                              end;
                            end;

                          end;

                        finally
                          Temp_TList.free;
                          iTunesBackup_Children_TList.free;
                        end;

                      end;
                    bl_Continue := False;
                    end;
                  end;
                end;
              finally
                if assigned(aPropertyTree) then
                begin
                  aPropertyTree.free;
                  aPropertyTree := nil;
                end;
              end;
            end;
          end;
        end;

        //------------------------------------------------------------------------
        // 7zip
        //------------------------------------------------------------------------
        if bl_Continue then
        begin
          if (anEntry.FileDriverInfo.ShortDisplayname = '7zip') then
          begin

            // The 7z already expanded test
            if anEntry.IsExpanded then
            begin
              FolderChildren_List := EntryList.Children(anEntry, True);
              try
                if FolderChildren_List.Count = 0 then
                begin
                  AddTo_Audit_Bucket(anEntry, isEncryptedTest_StringList, '7z already expanded test');
                  Encrypted_TUniqueList.Add(anEntry);
                  bl_Continue := False;
                end
              finally
                FolderChildren_List.free;
              end;
            end;

            // The 7z peek inside test
            if bl_Continue then
            begin
              anExpanded7zList := TList.Create;
              try
                //TaskProgress := NewProgress(True);
                TaskProgress := Progress.NewChild;
                try
                  ExtractCompoundFile(anEntry, anExpanded7zList, TaskProgress);
                except
                  Progress.Log(anEntry.EntryName + 'Exception extracting 7zip data: ' + anEntry.EntryName);
                end;
                Sleep(1000);
                Progress.DisplayTitle := SCRIPT_NAME;  // Refresh the title after the expand
                TaskProgress.DisplayMessages := ('Searching for encrypted files (Found: ' + Found_Count_str + ')' + RUNNING);
                Progress.DisplayMessages := ('Searching for encrypted files (Found: ' + Found_Count_str + ')' + RUNNING);

                if anExpanded7zList.Count = 0 then
                begin
                 CalculateEntropy(anEntry);
                  AddTo_Audit_Bucket(anEntry, isEncryptedTest_StringList, '7z peek inside test');
                  Encrypted_TUniqueList.Add(anEntry);
                  bl_Continue := False;
                end;

              finally
                anExpanded7zList.free;
              end;

            end;
          end;
        end;

        //------------------------------------------------------------------------
        // Documents - DOCX, PPTX, XLSX
        //------------------------------------------------------------------------
        if bl_Continue then
        begin
          upper_shortdisplayname_str := UpperCase(anEntry.FileDriverInfo.ShortDisplayname);
          if ((upper_shortdisplayname_str = 'XLSX') or
             (upper_shortdisplayname_str = 'DOCX') or
             (upper_shortdisplayname_str = 'PPTX') or
             (upper_shortdisplayname_str = MSENCRYPTED)) then
          begin
            DetermineSig(anEntry);
            if anEntry.isEncrypted then
            begin
              Encrypted_TUniqueList.Add(anEntry);
              CalculateEntropy(anEntry);
              AddTo_Audit_Bucket(anEntry, isEncryptedTest_StringList, 'isEncrypted Office XML');
              bl_Continue := False;
            end;
          end;
        end;

        //------------------------------------------------------------------------
        // isEncrypted - Other
        //------------------------------------------------------------------------
        if bl_Continue then
        begin
          upper_shortdisplayname_str := UpperCase(anEntry.FileDriverInfo.ShortDisplayname);
          if ((upper_shortdisplayname_str <> 'XLSX') and
             (upper_shortdisplayname_str <> 'DOCX') and
             (upper_shortdisplayname_str <> 'PPTX') and
             (upper_shortdisplayname_str <> MSENCRYPTED)) then
          begin
            if anEntry.isEncrypted then
            begin
              Encrypted_TUniqueList.Add(anEntry);
              CalculateEntropy(anEntry);
              AddTo_Audit_Bucket(anEntry, isEncryptedTest_StringList, 'isEncrypted Other');
              bl_Continue := False;
            end;
          end;
        end;

        //------------------------------------------------------------------------
        // Blackberry REM
        //------------------------------------------------------------------------
        if bl_Continue then
        begin
          if (UpperCase(anEntry.EntryNameExt) = '.REM') then
          begin
            Encrypted_TUniqueList.Add(anEntry);
            CalculateEntropy(anEntry);
            AddTo_Audit_Bucket(anEntry, isEncryptedTest_StringList, 'Blackberry .REM');
            bl_Continue := False;
          end;
        end;

        //------------------------------------------------------------------------
        // Check Archive Files
        //------------------------------------------------------------------------
        if bl_Continue then
        begin
          if (dtArchive in anEntry.FileDriverInfo.DriverType) then
          begin
            fileext := extractfileext(UpperCase(anEntry.EntryName));
            DetermineSig(anEntry);
            DeterminedFileDriverInfo := anEntry.DeterminedFileDriverInfo;
            DeterminedSig := DeterminedFileDriverInfo.ShortDisplayName;
            if (dtArchive in anEntry.DeterminedFileDriverInfo.DriverType) then
            begin
              // Exclude
              if (UpperCase(DeterminedSig) <> 'CAB') and
                 (UpperCase(fileext) <> '.IPA') and
                 (UpperCase(fileext) <> '.JAR') and
                 (UpperCase(fileext) <> '.IPSW') then
              begin

                //------------------------------------------------------------------
                // Check for "Encrypted" in the metadata nodes - Currently checks ZIP and RAR v4, v5
                //------------------------------------------------------------------
                if bl_Continue then
                begin
                  if (UpperCase(DeterminedSig) = 'ZIP') or (UpperCase(DeterminedSig) = 'RAR') then
                  begin
                    if aReader.OpenData(anEntry) then
                    begin
                      aPropertyTree := nil;
                      if ProcessMetadataProperties(anEntry, aPropertyTree, aReader) then
                      try
                        aRootProperty := aPropertyTree;
                        if assigned(aRootProperty) and assigned(aRootProperty.PropChildList) and (aRootProperty.PropChildList.Count >= 1) then
                        begin

                          aParentNode := TPropertyNode(aPropertyTree.GetFirstNodeByName('Encrypted')); //noslz
                          if assigned(aParentNode) then
                          begin
                            Encrypted_TUniqueList.Add(anEntry);
                            CalculateEntropy(anEntry);
                            if (UpperCase(DeterminedSig) = 'ZIP') then
                              AddTo_Audit_Bucket(anEntry, ZipMetadataTest_StringList, 'ZipMetadata');
                            if (UpperCase(DeterminedSig) = 'ZIP') then
                              AddTo_Audit_Bucket(anEntry, RAR5Test_StringList, 'RAR5Test');
                            bl_Continue := False;
                          end;

                        end;
                      finally
                        if assigned(aPropertyTree) then
                        begin
                          aPropertyTree.free;
                          aPropertyTree := nil;
                        end;
                      end;
                    end;
                  end;
                end;

                //------------------------------------------------------------------
                // RAR 5.0 Archive Format - https://www.rarlab.com/technote.htm
                //------------------------------------------------------------------
                if bl_Continue then
                begin
                  if UpperCase(DeterminedSig) = 'RAR' then
                  begin
                    Starting_Byte := 13;
                    if aReader.OpenData(anEntry) and (anEntry.LogicalSize > Starting_Byte) then
                    begin
                      ByteCount := aReader.Read(ByteArray, 1);
                      if ByteCount > 0 then
                      begin
                        aReader.Position := Starting_Byte;
                        hexbytes := aReader.AsHexString(ByteCount);
                        hexbytes := trim(hexbytes);
                        if hexbytes = '04' then
                        begin
                          Encrypted_TUniqueList.Add(anEntry);
                          CalculateEntropy(anEntry);
                          AddTo_Audit_Bucket(anEntry, RAR5Test_StringList, 'RAR5Test');
                          bl_Continue := False;
                        end;
                      end;
                    end;
                  end;
                end;

                fileEntropy := CalculateEntropy(anEntry);
                AddTo_Audit_Bucket(anEntry, NotDetected_StringList, 'NotDetected');

                //------------------------------------------------------------------
                // Archives - If aEntry reaches this point then calculate and test entropy
                //------------------------------------------------------------------
//                if bl_Continue then
//                begin
//                  if (UpperCase(DeterminedSig) <> 'ZIP') then
//                  begin
//                    if fileEntropy > 0.998 then
//                    begin
//                      HighEntropy_TList.Add(anEntry);
//                    end;
//                  end;
//                end;

              end;
            end;
          end; {Check Archives}
        end;

        Found_Count_str := IntToStr(Encrypted_TUniqueList.Count);
        Progress.DisplayMessage := ('Searching for encrypted files (Found: ' + Found_Count_str + ')' + RUNNING);

        Progress.IncCurrentprogress;
      end;
    finally
      aReader.free;
      EntryList.free;
      Working_TList.free;
    end;

    //============================================================================
    // Log Results
    //============================================================================
    gTotal_Count := 0;

    // Encrypted
    if assigned(Encrypted_TUniqueList) and (Encrypted_TUniqueList.Count > 0) then
    begin
      ProgressLogMemo('');
      ProgressLogMemo('The following FILES are encrypted (or contain encrypted files):');
      ProgressLogMemo('');
      ProgressLogMemo(format('%-6s %-10s %-10s %-100s', ['Count', 'Bates', 'Entropy', 'File Name']));
      ProgressLogMemo(format('%-6s %-10s %-10s %-100s', ['-----', '-----', '-------', '---------']));

      gTotal_Count := 0;
      i := 0;
      Enum := Encrypted_TUniqueList.GetEnumerator;
      while Enum.MoveNext do
      begin
        encEntry := Enum.Current;
        if encEntry.IsDirectory then
          Entropy_str := 'Folder'
        else
          Entropy_str := FieldEntropy.asString[encEntry];
        ProgressLogMemo(format('%-6s %-10s %-10s %-100s', [IntToStr(i + 1), IntToStr(encEntry.ID), Entropy_str, encEntry.FullPathName]));
        inc(i);
      end;

      gTotal_Count := gTotal_Count + Encrypted_TUniqueList.Count;
    end
    else
      ProgressLogMemo('No encrypted files found.');

    // Encrypted Warning PDF
    if  assigned(Encrypted_TUniqueList) and (Encrypted_TUniqueList.Count > 0) then
    begin
      Warning_PDF_StringList := TStringList.Create;
      try
        Heading_str := 'Encrypted Files (' + IntToStr(Encrypted_TUniqueList.Count) + ')';
        Warning_PDF_StringList.Add('Encrypted Files (' + IntToStr(Encrypted_TUniqueList.Count) + ')');
        Warning_PDF_StringList.Add(Stringofchar('-', Length(Heading_str)));
        Warning_PDF_StringList.Add(' ');

        Enum := Encrypted_TUniqueList.GetEnumerator;
		i := 0;
        while Enum.MoveNext do
        begin
          encEntry := Enum.Current;
          if encEntry.IsDirectory then
            Entropy_str := 'Folder'
          else
            Entropy_str := FieldEntropy.asString[encEntry];

          Warning_PDF_StringList.Add(format('%-6s %-100s',[IntToStr(i + 1) + '.', encEntry.FullPathName]));
		  Inc(i);
          Warning_PDF_StringList.Add(' ');
		  
		  if i >= 500 then
		  begin
		    Warning_PDF_StringList.Add(Stringofchar('=', 114));
		    Warning_PDF_StringList.Add('Stopped logging at 500 of ' + IntToStr(Encrypted_TUniqueList.Count) + ' records');
            Warning_PDF_StringList.Add(Stringofchar('=', 114));			
            break;            
		  end;			
		  
        end;

        CreatePDF(Warning_PDF_StringList, 'WARNING - Encrypted Files Found.pdf', 7); // number is font size

      finally
        Warning_PDF_StringList.free;
      end;
    end;

    // High Entropy
    if assigned(HighEntropy_TList) and (HighEntropy_TList.Count > 0) then
    begin
      for i := 0 to HighEntropy_TList.Count - 1 do
      begin
        encEntry := TEntry(HighEntropy_TList[i]);
        Entropy_str := FieldEntropy.asString[encEntry];
        if assigned(HighEntropy_StringList) then
          HighEntropy_StringList.Add(format('%-10s %-10s %-100s', [IntToStr(encEntry.ID), Entropy_str, encEntry.FullPathName]));
      end;
      gTotal_Count := gTotal_Count + HighEntropy_TList.Count;
    end;

    // Put into a StringList for sorting purposes
    if assigned(HighEntropy_StringList) and (HighEntropy_TList.Count > 0) then
    begin
      ProgressLogMemo('');
      ProgressLogMemo(Stringofchar('-', 80));
      ProgressLogMemo('');
      ProgressLogMemo('The following ARCHIVES have a high entropy score. Test for encryption:');
      ProgressLogMemo('');
      ProgressLogMemo(format('%-10s %-10s %-100s', [SPACE + 'Bates', 'Entropy', 'File Name']));
      ProgressLogMemo(format('%-10s %-10s %-100s', [SPACE + '-----', '-------', '---------']));
      HighEntropy_StringList.sort;
      for i := 0 to (HighEntropy_StringList.Count) - 1 do
      begin
        ProgressLogMemo(SPACE + HighEntropy_StringList(i));
      end;
    end;

    //------------------------------------------------------------------
    // Audit StringLists
    //------------------------------------------------------------------
    Progress.Log('');
    Progress.Log(Stringofchar('-', 80));
    Progress.Log('');
    Progress.Log('Audit:');
    Progress.Log(format('%-10s %-10s %-25s %-100s', [SPACE + 'Bates', 'Entropy', 'Reason', 'File Name']));
    Progress.Log(format('%-10s %-10s %-25s %-100s', [SPACE + '-----', '-------', '------', '---------']));

    if assigned(isEncryptedTest_StringList) then
    begin
      for i := 0 to (isEncryptedTest_StringList.Count) - 1 do
        Progress.Log(isEncryptedTest_StringList(i));
    end;

    if assigned(ZipMetadataTest_StringList) then
    begin
      for i := 0 to (ZipMetadataTest_StringList.Count) - 1 do
        Progress.Log(ZipMetadataTest_StringList(i));
    end;

    if assigned(RAR5Test_StringList) then
    begin
      for i := 0 to (RAR5Test_StringList.Count) - 1 do
        Progress.Log(RAR5Test_StringList(i));
    end;

    ProgressLogMemo('');
    ProgressLogMemo(Stringofchar('-', 80));
    ProgressLogMemo(SCRIPT_NAME + ' finished.');

    //------------------------------------------------------------------
    // Populate the OutList
    //------------------------------------------------------------------
    if assigned(Encrypted_TUniqueList) and (Encrypted_TUniqueList.Count > 0) then
    begin
      Enum := Encrypted_TUniqueList.GetEnumerator;
      while Enum.MoveNext do
      begin
        anEntry := Enum.Current;
        OutList.Add(anEntry);
      end;
    end;

  finally
    Encrypted_TUniqueList.free;
    FolderChildren_List.free;
    HighEntropy_StringList.free;
    HighEntropy_TList.free;
    isEncryptedTest_StringList.free;
    Memo_StringList.free;
    NotDetected_StringList.free;
    RAR5Test_StringList.free;
    ZipMetadataTest_StringList.free;
  end;
end;

//------------------------------------------------------------------------------
// Detect Encrypted Partitions
//------------------------------------------------------------------------------
procedure DetectEncryptedPartitions;
var
  anEntryList: TDataStore;
  anEntry: TEntry;
  FolderChildren_List: TList;
  aListEntry: TEntry;
  EncryptedPartition_StringList: TStringList;
  DecryptedPartition_StringList: TStringList;

begin
  anEntryList := GetDataStore(DATASTORE_FILESYSTEM);
  if not assigned(anEntryList) then
  begin
    Progress.Log('Not assigned a data store. The script will terminate.');
    Exit;
  end;

  anEntry := anEntryList.First;
  if not assigned(anEntry) then
  begin
    Progress.Log('No entries in the data store. The script will terminate.');
    anEntryList.free;
    Exit;
  end;

  EncryptedPartition_StringList := TStringList.Create;
  EncryptedPartition_StringList.Add('WARNING: Partition(s) appear to be encrypted:');
  EncryptedPartition_StringList.Add(Stringofchar('-', CHAR_LENGTH));

  DecryptedPartition_StringList := TStringList.Create;
  DecryptedPartition_StringList.Add('Decrypted Partition(s):');
  DecryptedPartition_StringList.Add(Stringofchar('-', CHAR_LENGTH));

  try
    Progress.Initialize(anEntryList.Count, 'Processing...');
    while assigned(anEntry) and (Progress.isRunning) do
    begin
      if (dstPartition in anEntry.Status) and (anEntry.isEncrypted) then
      begin
        FolderChildren_List := anEntryList.Children(anEntry, True); // True is fully recursive
        try
          // Encrypted Partition
          if assigned(FolderChildren_List) and (FolderChildren_List.Count = 1) then
          begin
            aListEntry := TEntry(FolderChildren_List.first);
            if assigned(aListEntry) and (aListEntry.isFreeSpace) then
            begin
              Progress.Log('Encrypted Partition: ' + anEntry.EntryName + SPACE + 'Size: ' + IntToStr(anEntry.LogicalSize));
              EncryptedPartition_StringList.Add(anEntry.EntryName + COMMA +
              'Size:' + SPACE + Format('%1.2n GB', [anEntry.PhysicalSize / (1024 * 1024 * 1024)]) + COMMA +
              'File Count:' + SPACE + IntToStr(FolderChildren_List.Count));
            end;
          end;

          // Decrypted Partition
          if assigned(FolderChildren_List) and (FolderChildren_List.Count > 2) then
          begin
            Progress.Log('Decrypted Partition: ' + anEntry.EntryName + SPACE + 'Size: ' + IntToStr(anEntry.LogicalSize));
            DecryptedPartition_StringList.Add(anEntry.EntryName + COMMA +
            'Size:' + SPACE + Format('%1.2n GB', [anEntry.PhysicalSize / (1024 * 1024 * 1024)]) + COMMA +
            'File Count:' + SPACE + IntToStr(FolderChildren_List.Count));
          end;

        finally
          FolderChildren_List.free;
        end;
      end;
      Progress.IncCurrentProgress;
      anEntry := anEntryList.Next;
    end;

    if assigned(EncryptedPartition_StringList) and (EncryptedPartition_StringList.Count > 2) then
      CreatePDF(EncryptedPartition_StringList, 'WARNING - Encrypted Drive.pdf', 8);

    if assigned(DecryptedPartition_StringList) and (DecryptedPartition_StringList.Count > 2) then
      CreatePDF(DecryptedPartition_StringList, 'INFORMATION - A drive was decrypted.pdf', 8);

  finally
    anEntryList.free;
    EncryptedPartition_StringList.free;
    DecryptedPartition_StringList.free;
  end;
  Progress.Log('Script finished.');
end;

//============================================================================
// Start of the Script
//============================================================================
begin
  Progress.DisplayTitle := SCRIPT_NAME;
  Progress.DisplayMessageNow := ('Starting' + RUNNING);
  Progress.Log(SCRIPT_NAME + SPACE + 'started' + RUNNING);

  ParametersReceived;

  // Check that InList and OutList exist, and InList count > 0
  if (not ListExists(InList, 'InList')) or (not ListExists(OutList, 'OutList')) then
  begin
    Progress.Log('Did not locate InList/OutList.' + SPACE + TSWT);
    Exit;
  end;

  MainProc;

  Progress.Log(Stringofchar('-', CHAR_LENGTH));
  Progress.Log(RPad('OutList on finish:', rpad_value) + IntToStr(OutList.Count));
  Progress.Log(SCRIPT_NAME + SPACE + 'finished.');
  
  if RegexMatch(ScriptTask.GetTaskListOption('OPENLOG'), 'True', False) then
    ShellExecute(0, nil, 'explorer.exe', pchar(Progress.LogFilename), nil, 1);
end.