﻿unit Export_Logical_Image_to_ZIP;

interface

uses
  Classes, Common, DataEntry, DataStorage, Graphics, Math, Regex, SysUtils;

const
  BS = '\';
  CHAR_LENGTH = 80;
  CR = #13#10;
  DCR = #13#10 + #13#10;
  HYPHEN = ' - ';
  OSFILENAME = '\\?\';
  RPAD_VALUE = 40;
  RUNNING = '...';
  SCRIPT_NAME = 'Export Logical Image to_ZIP';
  SPACE = ' ';
  TSWT = 'The script will terminate.';

var
  gCaseName_str: string;
  gDevice_TList: TList;
  gRunTimeReport_StringList: TStringList;

implementation

// ------------------------------------------------------------------------------
// Console Log
// ------------------------------------------------------------------------------
procedure ConsoleLog(aString: string);
begin
  if assigned(gRunTimeReport_StringList) then
    gRunTimeReport_StringList.Add(aString);
  Progress.Log(aString);
end;

// -----------------------------------------------------------------------------
// Export To Zip
// -----------------------------------------------------------------------------
procedure ExportToZip(aTList: TList; aFileName_str: string);
var
  TaskProgress: TPAC;
  bl_Continue: boolean;
  tick_export_Zip: uint64;
  export_filename_str: string;
begin
  if not Progress.isRunning then
    Exit;
  if not assigned(aTList) then
    Exit;
  export_filename_str := GetExportedDir + aFileName_str + '.Zip';
  if FileExists(export_filename_str) then
  export_filename_str := GetUniqueFileName(export_filename_str);
  tick_export_Zip := GetTickCount64;
  if assigned(aTList) and (aTList.Count > 0) then
  begin
    if Progress.isRunning then
    begin
      TaskProgress := NewProgress(true);
      RunTask('TCommandTask_ExportZip', DATASTORE_FILESYSTEM, aTList, TaskProgress, ['filename=' + export_filename_str]); // noslz
      while (TaskProgress.isRunning) and (Progress.isRunning) do
        Sleep(500);
      if not(Progress.isRunning) then
        TaskProgress.Cancel;
      bl_Continue := (TaskProgress.CompleteState = pcsSuccess);
    end;
    Sleep(500);
    if FileExists(export_filename_str) then
    begin
      ConsoleLog(RPad('Created:' + SPACE + ExtractFileName(export_filename_str), RPAD_VALUE) + GetZipSize(export_filename_str) + SPACE + '(' + IntToStr(aTList.Count) + ' files)');
      ConsoleLog(RPad('Time Taken:', RPAD_VALUE) + TimeTakenStr(tick_export_Zip));
    end;
  end;
end;

//------------------------------------------------------------------------------
// Get Device TList
//------------------------------------------------------------------------------
procedure GetDeviceList(aTList: TList);
var
  anEntry: TEntry;
  dEntry: TEntry;
  DataStore_EV: TDataStore;
  i: integer;
begin
  DataStore_EV := GetDataStore(DATASTORE_EVIDENCE);
  If not assigned(DataStore_EV) then
    Exit;
  try
    aTList.clear;
    anEntry := DataStore_EV.First;
    while assigned(anEntry) and (Progress.isRunning) do
    begin
      if anEntry.IsDevice then
        aTList.Add(anEntry);
      anEntry := DataStore_EV.next;
    end;
    ConsoleLog(RPad('Case:', RPAD_VALUE) + gCaseName_str);
    ConsoleLog(RPad('Devices:', RPAD_VALUE) + inttostr(aTList.Count));
    for i := 0 to aTList.Count - 1 do
    begin
      dEntry := TEntry(aTList[i]);
      ConsoleLog(RPad(HYPHEN + dEntry.EntryName, RPAD_VALUE) +  GetSizeStr(dEntry.LogicalSize));
    end;
    ConsoleLog(StringOfChar('-', CHAR_LENGTH));
  finally
    DataStore_EV.free;
  end;
end;

//------------------------------------------------------------------------------
// Get Size As String
//------------------------------------------------------------------------------
function GetSizeStr(asize: int64): string;
begin
  if asize >= 1073741824 then
    Result := (Format('%1.2n GB', [asize / (1024 * 1024 * 1024)]))
  else if (asize < 1073741824) and (asize > 1048576) then
    Result := (Format('%1.2n MB', [asize / (1024 * 1024)]))
  else if asize < 1048576 then
    Result := (Format('%1.0n KB', [asize / (1024)]));
end;

// -----------------------------------------------------------------------------
// Procedure: List files in folder as string
// -----------------------------------------------------------------------------
function GetZipSize(file_path_str: string): string;
var
  filesize_int64: int64;
  i: integer;
  ListOfFiles_StringList: TStringList;
begin
  Result := '';
  filesize_int64 := 0;
  ListOfFiles_StringList := TStringList.Create;
  try
    if DirectoryExists(ExtractFileDir(file_path_str)) then
    begin
      ListOfFiles_StringList := GetAllFileList(ExtractFileDir(file_path_str), '*.Zip', False, Progress);
      if assigned(ListOfFiles_StringList) then
      begin
        for i := 0 to ListOfFiles_StringList.Count - 1 do
        begin
          if ListOfFiles_StringList[i] = file_path_str then
          begin
            filesize_int64 := GetFileSize(ListOfFiles_StringList[i]);
            Break;
          end;  
        end;
      end;
    end;
    if filesize_int64 > 0 then
      Result := GetSizeStr(filesize_int64);
  finally
    ListOfFiles_StringList.free;
  end;
end;

// -----------------------------------------------------------------------------
// RPad
// -----------------------------------------------------------------------------
function RPad(const AString: string; AChars: integer): string;
begin
  AChars := AChars - Length(AString) + 1;
  if AChars > 0 then
    Result := AString + StringOfChar(' ', AChars)
  else
    Result := AString;
end;

//------------------------------------------------------------------------------
// Create RunTime PDF
//------------------------------------------------------------------------------
procedure CreateRunTimePDF(aStringList: TStringList; save_name_str: string; font_size_int: integer; DT_In_FileName_bl: boolean);
const
  HYPHEN = ' - ';
var
  CurrentCaseDir: string;
  ExportDate: string;
  ExportDateTime: string;
  ExportedDir: string;
  ExportTime: string;
  SaveName: string;
begin
  sleep(1000);
  CurrentCaseDir := GetCurrentCaseDir;
  ExportedDir := CurrentCaseDir + 'Reports\Run Time Reports\';
  ExportDate := formatdatetime('yyyy-mm-dd', now);
  ExportTime := formatdatetime('hh-nn-ss', now);
  if DT_In_FileName_bl then
    ExportDateTime := (ExportDate + HYPHEN + ExportTime + HYPHEN)
  else
    ExportDateTime := '';
  SaveName := ExportedDir + ExportDateTime  + save_name_str;
  if ExtractFileExt(SaveName) = '' then
    SaveName := SaveName + '.pdf';
  if DirectoryExists(CurrentCaseDir) then
  begin
    try
      if ForceDirectories(IncludeTrailingPathDelimiter(ExportedDir)) and DirectoryExists(ExportedDir) then
      begin
        if GeneratePDF(SaveName, aStringList, font_size_int) then
          ConsoleLog('Success Creating: ' + SaveName)
      end
      else
        ConsoleLog('Failed Creating: ' + SaveName);
    except
      ConsoleLog('Exception creating runtime folder');
    end;
  end
  else
    ConsoleLog('There is no current case folder');
end;

// -----------------------------------------------------------------------------
// Time Taken - As String (hh:mm:ss)
// -----------------------------------------------------------------------------
function TimeTakenStr(the_tick: uint64): string;
var
  time_taken_tick: uint64;
  time_taken_int: uint64;
begin
  Result := '';
  time_taken_tick := (GetTickCount - the_tick);
  time_taken_int := trunc(time_taken_tick / 1000);
  Result := FormatDateTime('hh:mm:ss', time_taken_int / SecsPerDay);
end;

// -----------------------------------------------------------------------------
// Unique Filename
// -----------------------------------------------------------------------------
function GetUniqueFileName(filename_str: string): string;
var
  fnum: integer;
  fname: string;
  fpath: string;
  fext: string;
begin
  Result := filename_str;
  fext  := ExtractFileExt(filename_str);
  fnum  := 1;
  fname := ExtractFileName(filename_str);
  fname := ChangeFileExt(fname, '');
  fpath := ExtractFilePath(filename_str);
  while FileExists(fpath + fname + '_' + IntToStr(fnum) + fext) do
    fnum := fnum + 1;
  Result := fpath + fname + '_' + IntToStr(fnum) + fext;
end;

// =============================================================================
// Start of Script
// =============================================================================
var
  anEntry: TEntry;
  DataStore_FS: TDataStore;
  i: integer;
  Children_TList: TList;
  zip_name_str: string;
  firstEntry: TEntry;

begin
  ConsoleLog(StringOfChar('-', CHAR_LENGTH));
  DataStore_FS := GetDataStore(DATASTORE_FILESYSTEM);
  If not assigned(DataStore_FS) then
  begin
    ConsoleLog('Not assigned Datastore_FS.' + SPACE + TSWT);
    Exit;
  end;

  try
    if DataStore_FS.Count > 0 then
    begin
      firstEntry := DataStore_FS.First;
      gCaseName_str := firstEntry.EntryName;
    end
    else
    begin
      ConsoleLog('No files in the File System module.' + SPACE + TSWT);
      Exit;
    end;
    
    gDevice_TList := TList.Create;
    gRunTimeReport_StringList := TStringList.Create;
    try
      GetDeviceList(gDevice_TList);

      for i := 0 to gDevice_TList.Count - 1 do
      begin
        anEntry := TEntry(gDevice_TList[i]);
        if RegexMatch(UpperCase(anEntry.Extension),'\.(L01|LX01|AD1)', True) then
        begin
          Children_TList := DataStore_FS.Children(anEntry, True); // True or False for subfolders
          try
            if assigned(Children_TList) and (Children_TList.Count > 0) then
            begin
              ConsoleLog(RPad('Exporting To Zip:', RPAD_VALUE) + anEntry.EntryName);
              zip_name_str := anEntry.EntryName;
              zip_name_str := StringReplace(zip_name_str, anEntry.Extension, '', [rfReplaceAll]);
              ExportToZip(Children_TList, zip_name_str);
              ConsoleLog(StringOfChar('-', CHAR_LENGTH));
            end;
          finally
            Children_TList.free;
          end;
        end;
      end;
      
      ConsoleLog('Script finished.');
      CreateRunTimePDF(gRunTimeReport_StringList, 'Process Summary', 8, False); // True is DT in Filename.PDF

    finally
      gDevice_TList.free;
      gRunTimeReport_StringList.free;
    end;

  finally
    DataStore_FS.free;
  end;
end.
